<?php
/**
 * Two-Factor Authentication (2FA)
 * TOTP support for Google Authenticator, Authy, etc.
 *
 * @package ArtistPro_Security
 * @since 1.1.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_Two_Factor_Auth {

    /**
     * Initialize 2FA
     */
    public static function init() {
        // Add 2FA field to login form
        add_action('login_form', array(__CLASS__, 'add_2fa_field'));
        
        // Validate 2FA code
        add_filter('authenticate', array(__CLASS__, 'validate_2fa'), 50, 3);
        
        // User profile settings
        add_action('show_user_profile', array(__CLASS__, 'user_profile_fields'));
        add_action('edit_user_profile', array(__CLASS__, 'user_profile_fields'));
        add_action('personal_options_update', array(__CLASS__, 'save_user_profile'));
        add_action('edit_user_profile_update', array(__CLASS__, 'save_user_profile'));
        
        // AJAX handlers
        add_action('wp_ajax_generate_2fa_secret', array(__CLASS__, 'ajax_generate_secret'));
        add_action('wp_ajax_verify_2fa_code', array(__CLASS__, 'ajax_verify_code'));
        add_action('wp_ajax_disable_2fa', array(__CLASS__, 'ajax_disable_2fa'));
        add_action('wp_ajax_generate_backup_codes', array(__CLASS__, 'ajax_generate_backup_codes'));
        
        // Admin settings
        add_action('admin_init', array(__CLASS__, 'register_settings'));
    }

    /**
     * Generate TOTP secret key
     *
     * @return string Base32 encoded secret
     */
    public static function generate_secret() {
        $secret = '';
        $chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567'; // Base32 alphabet
        
        for ($i = 0; $i < 16; $i++) {
            $secret .= $chars[random_int(0, 31)];
        }
        
        return $secret;
    }

    /**
     * Generate TOTP code
     *
     * @param string $secret Base32 secret key
     * @param int $time_slice Time slice (default: current time)
     * @return string 6-digit code
     */
    public static function get_code($secret, $time_slice = null) {
        if ($time_slice === null) {
            $time_slice = floor(time() / 30);
        }
        
        $secret = self::base32_decode($secret);
        
        // Pack time into binary string
        $time = pack('N*', 0) . pack('N*', $time_slice);
        
        // Hash it with HMAC-SHA1
        $hash = hash_hmac('sha1', $time, $secret, true);
        
        // Extract dynamic binary code
        $offset = ord(substr($hash, -1)) & 0x0F;
        $code = (
            ((ord($hash[$offset + 0]) & 0x7F) << 24) |
            ((ord($hash[$offset + 1]) & 0xFF) << 16) |
            ((ord($hash[$offset + 2]) & 0xFF) << 8) |
            (ord($hash[$offset + 3]) & 0xFF)
        ) % 1000000;
        
        return str_pad($code, 6, '0', STR_PAD_LEFT);
    }

    /**
     * Verify TOTP code
     *
     * @param string $secret Base32 secret key
     * @param string $code User-provided code
     * @param int $discrepancy Time window (±30 seconds per step)
     * @return bool
     */
    public static function verify_code($secret, $code, $discrepancy = 1) {
        $current_time_slice = floor(time() / 30);
        
        // Check current time and nearby time slices
        for ($i = -$discrepancy; $i <= $discrepancy; $i++) {
            $calculated_code = self::get_code($secret, $current_time_slice + $i);
            
            if (hash_equals($calculated_code, $code)) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Generate QR code URL for Google Authenticator
     *
     * @param string $secret Secret key
     * @param string $name Account name
     * @param string $issuer Issuer name
     * @return string QR code URL
     */
    public static function get_qr_code_url($secret, $name, $issuer = 'ArtistPro Security') {
        $issuer = rawurlencode($issuer);
        $name = rawurlencode($name);
        
        $otpauth = "otpauth://totp/{$issuer}:{$name}?secret={$secret}&issuer={$issuer}";
        
        // Use Google Charts API for QR code
        return 'https://chart.googleapis.com/chart?chs=200x200&chld=M|0&cht=qr&chl=' . urlencode($otpauth);
    }

    /**
     * Generate backup codes
     *
     * @param int $count Number of codes to generate
     * @return array
     */
    public static function generate_backup_codes($count = 10) {
        $codes = array();
        
        for ($i = 0; $i < $count; $i++) {
            $code = '';
            for ($j = 0; $j < 4; $j++) {
                $code .= str_pad(random_int(0, 9999), 4, '0', STR_PAD_LEFT);
                if ($j < 3) {
                    $code .= '-';
                }
            }
            $codes[] = $code;
        }
        
        return $codes;
    }

    /**
     * Check if 2FA is enabled for user
     *
     * @param int $user_id User ID
     * @return bool
     */
    public static function is_enabled_for_user($user_id) {
        $enabled = get_user_meta($user_id, 'artistpro_2fa_enabled', true);
        return (bool) $enabled;
    }

    /**
     * Check if 2FA is required for user's role
     *
     * @param WP_User $user User object
     * @return bool
     */
    public static function is_required_for_user($user) {
        $required_roles = get_option('artistpro_security_2fa_required_roles', array('administrator'));
        
        if (empty($required_roles)) {
            return false;
        }
        
        foreach ($user->roles as $role) {
            if (in_array($role, $required_roles)) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Add 2FA code field to login form
     */
    public static function add_2fa_field() {
        ?>
        <p class="artistpro-2fa-field" style="display:none;">
            <label for="artistpro_2fa_code">
                <?php _e('Two-Factor Code', 'artistpro-security'); ?>
                <br>
                <input type="text" name="artistpro_2fa_code" id="artistpro_2fa_code" 
                       class="input" value="" size="20" 
                       autocomplete="off" inputmode="numeric" pattern="[0-9]*"
                       placeholder="000000" maxlength="20">
            </label>
        </p>
        <script>
        // Show 2FA field if username has 2FA enabled (we'll check via AJAX in production)
        // For now, always show if user enters username
        </script>
        <?php
    }

    /**
     * Validate 2FA code during login
     *
     * @param WP_User|WP_Error|null $user User object or error
     * @param string $username Username
     * @param string $password Password
     * @return WP_User|WP_Error
     */
    public static function validate_2fa($user, $username, $password) {
        // Skip if already an error
        if (is_wp_error($user)) {
            return $user;
        }
        
        // Skip if not a valid user
        if (!($user instanceof WP_User)) {
            return $user;
        }
        
        // Check if 2FA is enabled for this user
        if (!self::is_enabled_for_user($user->ID)) {
            // Check if 2FA is required for this user's role
            if (self::is_required_for_user($user)) {
                return new WP_Error(
                    'artistpro_2fa_required',
                    __('Two-Factor Authentication is required for your account. Please configure it in your profile.', 'artistpro-security')
                );
            }
            
            return $user;
        }
        
        // Get submitted 2FA code
        $submitted_code = isset($_POST['artistpro_2fa_code']) ? sanitize_text_field($_POST['artistpro_2fa_code']) : '';
        
        if (empty($submitted_code)) {
            return new WP_Error(
                'artistpro_2fa_code_required',
                __('Please enter your two-factor authentication code.', 'artistpro-security')
            );
        }
        
        // Remove any spaces or dashes
        $submitted_code = str_replace(array(' ', '-'), '', $submitted_code);
        
        // Check if it's a backup code
        if (strlen($submitted_code) > 6) {
            if (self::verify_backup_code($user->ID, $submitted_code)) {
                return $user;
            }
        }
        
        // Verify TOTP code
        $secret = get_user_meta($user->ID, 'artistpro_2fa_secret', true);
        
        if (empty($secret)) {
            return new WP_Error(
                'artistpro_2fa_not_configured',
                __('Two-Factor Authentication is not properly configured.', 'artistpro-security')
            );
        }
        
        if (!self::verify_code($secret, $submitted_code)) {
            // Log failed 2FA attempt
            ArtistPro_Login_Logger::log_attempt(array(
                'ip_address' => ArtistPro_IP_Manager::get_client_ip(),
                'username' => $username,
                'login_type' => 'wp_login',
                'result' => 'failed',
                'failure_reason' => 'Invalid 2FA code',
            ));
            
            return new WP_Error(
                'artistpro_2fa_invalid_code',
                __('Invalid two-factor authentication code.', 'artistpro-security')
            );
        }
        
        return $user;
    }

    /**
     * Verify backup code
     *
     * @param int $user_id User ID
     * @param string $code Backup code
     * @return bool
     */
    private static function verify_backup_code($user_id, $code) {
        $backup_codes = get_user_meta($user_id, 'artistpro_2fa_backup_codes', true);
        
        if (!is_array($backup_codes)) {
            return false;
        }
        
        // Hash the submitted code
        $code_hash = hash('sha256', $code);
        
        // Check if hash exists in backup codes
        $key = array_search($code_hash, $backup_codes);
        
        if ($key !== false) {
            // Remove used backup code
            unset($backup_codes[$key]);
            update_user_meta($user_id, 'artistpro_2fa_backup_codes', $backup_codes);
            
            return true;
        }
        
        return false;
    }

    /**
     * Add 2FA settings to user profile
     *
     * @param WP_User $user User object
     */
    public static function user_profile_fields($user) {
        if (!current_user_can('edit_user', $user->ID)) {
            return;
        }
        
        $enabled = self::is_enabled_for_user($user->ID);
        $secret = get_user_meta($user->ID, 'artistpro_2fa_secret', true);
        $backup_codes = get_user_meta($user->ID, 'artistpro_2fa_backup_codes', true);
        $required = self::is_required_for_user($user);
        
        ?>
        <h2><?php _e('Two-Factor Authentication', 'artistpro-security'); ?></h2>
        <table class="form-table">
            <tr>
                <th><?php _e('Status', 'artistpro-security'); ?></th>
                <td>
                    <?php if ($enabled): ?>
                        <span style="color: green; font-weight: bold;">✅ Enabled</span>
                        <p>
                            <button type="button" class="button" id="artistpro-disable-2fa">
                                Disable Two-Factor Authentication
                            </button>
                            <button type="button" class="button" id="artistpro-regenerate-backup-codes">
                                Generate New Backup Codes
                            </button>
                        </p>
                    <?php else: ?>
                        <span style="color: <?php echo $required ? 'red' : 'gray'; ?>; font-weight: bold;">
                            <?php echo $required ? '⚠️ Required' : '❌ Disabled'; ?>
                        </span>
                        <?php if ($required): ?>
                            <p style="color: red;">
                                <strong>Two-Factor Authentication is required for your role.</strong>
                            </p>
                        <?php endif; ?>
                        <p>
                            <button type="button" class="button button-primary" id="artistpro-enable-2fa">
                                Enable Two-Factor Authentication
                            </button>
                        </p>
                    <?php endif; ?>
                    
                    <div id="artistpro-2fa-setup" style="display:none; margin-top: 20px; padding: 20px; background: #f9f9f9; border: 1px solid #ddd;">
                        <h3>Setup Two-Factor Authentication</h3>
                        <p>Scan this QR code with Google Authenticator, Authy, or another TOTP app:</p>
                        <div id="artistpro-2fa-qrcode"></div>
                        <p>Or manually enter this secret key: <code id="artistpro-2fa-secret"></code></p>
                        
                        <p>
                            <label>
                                <strong>Enter code from your app to verify:</strong><br>
                                <input type="text" id="artistpro-2fa-verify-code" 
                                       placeholder="000000" maxlength="6" 
                                       style="font-size: 18px; width: 150px; text-align: center;">
                            </label>
                            <button type="button" class="button button-primary" id="artistpro-2fa-verify-button">
                                Verify & Enable
                            </button>
                        </p>
                        <div id="artistpro-2fa-status"></div>
                    </div>
                    
                    <div id="artistpro-2fa-backup-codes" style="display:none; margin-top: 20px; padding: 20px; background: #fffbcc; border: 1px solid #e6db55;">
                        <h3>⚠️ Backup Codes</h3>
                        <p><strong>Save these codes in a safe place!</strong> Each can be used once if you lose access to your authenticator app.</p>
                        <div id="artistpro-backup-codes-list" style="font-family: monospace; font-size: 14px;"></div>
                        <p>
                            <button type="button" class="button" onclick="window.print();">Print Codes</button>
                            <button type="button" class="button" id="artistpro-copy-backup-codes">Copy to Clipboard</button>
                        </p>
                    </div>
                </td>
            </tr>
        </table>
        
        <script>
        jQuery(document).ready(function($) {
            $('#artistpro-enable-2fa').on('click', function() {
                var $btn = $(this);
                $btn.prop('disabled', true).text('Generating...');
                
                $.post(ajaxurl, {
                    action: 'generate_2fa_secret',
                    user_id: <?php echo $user->ID; ?>,
                    _wpnonce: '<?php echo wp_create_nonce('artistpro_2fa_' . $user->ID); ?>'
                }, function(response) {
                    if (response.success) {
                        $('#artistpro-2fa-secret').text(response.data.secret);
                        $('#artistpro-2fa-qrcode').html('<img src="' + response.data.qr_url + '" alt="QR Code">');
                        $('#artistpro-2fa-setup').show();
                        $btn.hide();
                    } else {
                        alert('Error: ' + response.data);
                        $btn.prop('disabled', false).text('Enable Two-Factor Authentication');
                    }
                });
            });
            
            $('#artistpro-2fa-verify-button').on('click', function() {
                var code = $('#artistpro-2fa-verify-code').val();
                var $status = $('#artistpro-2fa-status');
                
                if (!code || code.length !== 6) {
                    $status.html('<p style="color:red;">Please enter a 6-digit code.</p>');
                    return;
                }
                
                $.post(ajaxurl, {
                    action: 'verify_2fa_code',
                    user_id: <?php echo $user->ID; ?>,
                    code: code,
                    _wpnonce: '<?php echo wp_create_nonce('artistpro_2fa_' . $user->ID); ?>'
                }, function(response) {
                    if (response.success) {
                        $status.html('<p style="color:green;">✅ Verified! Two-Factor Authentication is now enabled.</p>');
                        
                        // Show backup codes
                        var codes = response.data.backup_codes;
                        var html = '';
                        codes.forEach(function(code) {
                            html += code + '<br>';
                        });
                        $('#artistpro-backup-codes-list').html(html);
                        $('#artistpro-2fa-backup-codes').show();
                        
                        // Reload page after 3 seconds
                        setTimeout(function() {
                            location.reload();
                        }, 3000);
                    } else {
                        $status.html('<p style="color:red;">❌ ' + response.data + '</p>');
                    }
                });
            });
            
            $('#artistpro-disable-2fa').on('click', function() {
                if (!confirm('Are you sure you want to disable Two-Factor Authentication?')) {
                    return;
                }
                
                $.post(ajaxurl, {
                    action: 'disable_2fa',
                    user_id: <?php echo $user->ID; ?>,
                    _wpnonce: '<?php echo wp_create_nonce('artistpro_2fa_' . $user->ID); ?>'
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert('Error: ' + response.data);
                    }
                });
            });
        });
        </script>
        <?php
    }

    /**
     * Save user profile 2FA settings
     *
     * @param int $user_id User ID
     */
    public static function save_user_profile($user_id) {
        // Settings are saved via AJAX
    }

    /**
     * Base32 decode
     *
     * @param string $secret Base32 encoded string
     * @return string
     */
    private static function base32_decode($secret) {
        $base32chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ234567';
        $base32charsFlipped = array_flip(str_split($base32chars));
        
        $paddingCharCount = substr_count($secret, '=');
        $allowedValues = array(6, 4, 3, 1, 0);
        
        if (!in_array($paddingCharCount, $allowedValues)) {
            return false;
        }
        
        for ($i = 0; $i < 4; $i++) {
            if ($paddingCharCount == $allowedValues[$i] &&
                substr($secret, -($allowedValues[$i])) != str_repeat('=', $allowedValues[$i])) {
                return false;
            }
        }
        
        $secret = str_replace('=', '', $secret);
        $secret = str_split($secret);
        $binaryString = '';
        
        for ($i = 0; $i < count($secret); $i = $i + 8) {
            $x = '';
            if (!in_array($secret[$i], $base32charsFlipped)) {
                return false;
            }
            for ($j = 0; $j < 8; $j++) {
                $x .= str_pad(base_convert(@$base32charsFlipped[@$secret[$i + $j]], 10, 2), 5, '0', STR_PAD_LEFT);
            }
            $eightBits = str_split($x, 8);
            for ($z = 0; $z < count($eightBits); $z++) {
                $binaryString .= (($y = chr(base_convert($eightBits[$z], 2, 10))) || ord($y) == 48) ? $y : '';
            }
        }
        
        return $binaryString;
    }

    /**
     * AJAX: Generate 2FA secret
     */
    public static function ajax_generate_secret() {
        $user_id = isset($_POST['user_id']) ? (int) $_POST['user_id'] : 0;
        
        check_ajax_referer('artistpro_2fa_' . $user_id);
        
        if (!current_user_can('edit_user', $user_id)) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $secret = self::generate_secret();
        $user = get_userdata($user_id);
        
        // Store secret temporarily (not enabled yet)
        update_user_meta($user_id, 'artistpro_2fa_secret_temp', $secret);
        
        $qr_url = self::get_qr_code_url($secret, $user->user_login);
        
        wp_send_json_success(array(
            'secret' => $secret,
            'qr_url' => $qr_url,
        ));
    }

    /**
     * AJAX: Verify 2FA code and enable
     */
    public static function ajax_verify_code() {
        $user_id = isset($_POST['user_id']) ? (int) $_POST['user_id'] : 0;
        $code = isset($_POST['code']) ? sanitize_text_field($_POST['code']) : '';
        
        check_ajax_referer('artistpro_2fa_' . $user_id);
        
        if (!current_user_can('edit_user', $user_id)) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $secret = get_user_meta($user_id, 'artistpro_2fa_secret_temp', true);
        
        if (empty($secret)) {
            wp_send_json_error('No secret found. Please try again.');
        }
        
        if (!self::verify_code($secret, $code)) {
            wp_send_json_error('Invalid code. Please try again.');
        }
        
        // Code is valid - enable 2FA
        update_user_meta($user_id, 'artistpro_2fa_secret', $secret);
        update_user_meta($user_id, 'artistpro_2fa_enabled', true);
        delete_user_meta($user_id, 'artistpro_2fa_secret_temp');
        
        // Generate backup codes
        $backup_codes = self::generate_backup_codes();
        $backup_codes_hashed = array_map(function($code) {
            return hash('sha256', $code);
        }, $backup_codes);
        
        update_user_meta($user_id, 'artistpro_2fa_backup_codes', $backup_codes_hashed);
        
        wp_send_json_success(array(
            'message' => 'Two-Factor Authentication enabled successfully!',
            'backup_codes' => $backup_codes,
        ));
    }

    /**
     * AJAX: Disable 2FA
     */
    public static function ajax_disable_2fa() {
        $user_id = isset($_POST['user_id']) ? (int) $_POST['user_id'] : 0;
        
        check_ajax_referer('artistpro_2fa_' . $user_id);
        
        if (!current_user_can('edit_user', $user_id)) {
            wp_send_json_error('Insufficient permissions');
        }
        
        delete_user_meta($user_id, 'artistpro_2fa_enabled');
        delete_user_meta($user_id, 'artistpro_2fa_secret');
        delete_user_meta($user_id, 'artistpro_2fa_backup_codes');
        
        wp_send_json_success('Two-Factor Authentication disabled');
    }

    /**
     * AJAX: Generate new backup codes
     */
    public static function ajax_generate_backup_codes() {
        $user_id = isset($_POST['user_id']) ? (int) $_POST['user_id'] : 0;
        
        check_ajax_referer('artistpro_2fa_' . $user_id);
        
        if (!current_user_can('edit_user', $user_id)) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $backup_codes = self::generate_backup_codes();
        $backup_codes_hashed = array_map(function($code) {
            return hash('sha256', $code);
        }, $backup_codes);
        
        update_user_meta($user_id, 'artistpro_2fa_backup_codes', $backup_codes_hashed);
        
        wp_send_json_success(array(
            'backup_codes' => $backup_codes,
        ));
    }

    /**
     * Register settings
     */
    public static function register_settings() {
        register_setting('artistpro_security_2fa', 'artistpro_security_2fa_required_roles');
    }
}
