<?php
/**
 * Security Notifier - Email notifications
 *
 * @package ArtistPro_Security
 * @since 1.0.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_Security_Notifier {

    /**
     * @var string Notification email address
     */
    private $notify_email;

    /**
     * Constructor
     */
    public function __construct() {
        $this->notify_email = get_option('artistpro_security_notify_email', get_option('admin_email'));
    }

    /**
     * Initialize hooks
     */
    public function init() {
        // Hook into lockout events
        add_action('artistpro_security_lockout', [$this, 'on_lockout'], 10, 2);
        add_action('artistpro_security_permanent_block', [$this, 'on_permanent_block'], 10, 2);
    }

    /**
     * Static method to send lockout notification
     * Called from ArtistPro_Security_Core::trigger_lockout()
     *
     * @param string $ip IP address
     * @param string $username Username that triggered lockout
     * @param int $failures Number of failed attempts
     */
    public static function send_lockout_notification($ip, $username, $failures) {
        $instance = new self();
        $instance->on_lockout($ip, array(
            'username' => $username,
            'attempts_count' => $failures,
            'lockout_type' => 'short',
            'locked_until' => date('Y-m-d H:i:s', time() + 1200), // 20 min default
        ));
    }

    /**
     * Handle lockout notification
     *
     * @param string $ip IP address
     * @param array $data Lockout data
     */
    public function on_lockout($ip, $data) {
        if (!get_option('artistpro_security_notify_lockouts', true)) {
            return;
        }

        // Rate limit notifications (max 1 per IP per hour)
        $cache_key = 'artistpro_security_notified_' . md5($ip);
        if (get_transient($cache_key)) {
            return;
        }
        set_transient($cache_key, true, HOUR_IN_SECONDS);

        $subject = sprintf('[%s] Login lockout: %s', get_bloginfo('name'), $ip);

        $message = $this->get_lockout_email_html($ip, $data);

        $this->send_email($subject, $message);
    }

    /**
     * Handle permanent block notification
     *
     * @param string $ip IP address
     * @param string $reason Reason for block
     */
    public function on_permanent_block($ip, $reason) {
        if (!get_option('artistpro_security_notify_blocks', true)) {
            return;
        }

        $subject = sprintf('[%s] IP permanently blocked: %s', get_bloginfo('name'), $ip);

        $message = $this->get_block_email_html($ip, $reason);

        $this->send_email($subject, $message);
    }

    /**
     * Send daily summary
     *
     * @param object $stats Daily statistics
     * @param array $top_blocked Top blocked IPs
     */
    public function send_daily_summary($stats, $top_blocked) {
        $subject = sprintf('[%s] Security Daily Summary', get_bloginfo('name'));

        $message = $this->get_summary_email_html($stats, $top_blocked);

        $this->send_email($subject, $message);
    }

    /**
     * Send email
     *
     * @param string $subject Email subject
     * @param string $message HTML message
     * @return bool
     */
    private function send_email($subject, $message) {
        $headers = [
            'Content-Type: text/html; charset=UTF-8',
            'From: artistpro Security <noreply@' . parse_url(home_url(), PHP_URL_HOST) . '>'
        ];

        return wp_mail($this->notify_email, $subject, $message, $headers);
    }

    /**
     * Get lockout email HTML
     *
     * @param string $ip IP address
     * @param array $data Lockout data
     * @return string
     */
    private function get_lockout_email_html($ip, $data) {
        $lockout_type = $data['lockout_type'] ?? 'short';
        $attempts = $data['attempts_count'] ?? 0;
        $locked_until = $data['locked_until'] ?? '';
        $username = $data['username'] ?? 'Unknown';

        // Get GeoIP info
        $geo_blocker = new ArtistPro_Geo_Blocker();
        $country = $geo_blocker->get_country($ip);
        $country_name = $country ? $country['name'] : 'Unknown';

        $duration = $lockout_type === 'long' ? '24 hours' : '20 minutes';

        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <style>
                body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #dc3545; color: white; padding: 20px; border-radius: 8px 8px 0 0; }
                .content { background: #f8f9fa; padding: 20px; border-radius: 0 0 8px 8px; }
                .info-table { width: 100%; border-collapse: collapse; margin: 15px 0; }
                .info-table td { padding: 10px; border-bottom: 1px solid #dee2e6; }
                .info-table td:first-child { font-weight: bold; width: 140px; }
                .footer { text-align: center; color: #6c757d; font-size: 12px; margin-top: 20px; }
                .btn { display: inline-block; padding: 10px 20px; background: #007bff; color: white; text-decoration: none; border-radius: 4px; }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h2 style="margin:0;">🔒 Login Lockout Alert</h2>
                </div>
                <div class="content">
                    <p>An IP address has been locked out due to repeated failed login attempts.</p>

                    <table class="info-table">
                        <tr><td>IP Address</td><td><strong><?php echo esc_html($ip); ?></strong></td></tr>
                        <tr><td>Country</td><td><?php echo esc_html($country_name); ?></td></tr>
                        <tr><td>Username Tried</td><td><?php echo esc_html($username); ?></td></tr>
                        <tr><td>Failed Attempts</td><td><?php echo esc_html($attempts); ?></td></tr>
                        <tr><td>Lockout Duration</td><td><?php echo esc_html($duration); ?></td></tr>
                        <tr><td>Locked Until</td><td><?php echo esc_html($locked_until); ?></td></tr>
                    </table>

                    <p>
                        <a href="<?php echo admin_url('options-general.php?page=artistpro-security&tab=logs'); ?>" class="btn">
                            View Security Logs
                        </a>
                    </p>
                </div>
                <div class="footer">
                    <p>This notification was sent by artistpro Security on <?php echo get_bloginfo('name'); ?></p>
                </div>
            </div>
        </body>
        </html>
        <?php
        return ob_get_clean();
    }

    /**
     * Get permanent block email HTML
     *
     * @param string $ip IP address
     * @param string $reason Block reason
     * @return string
     */
    private function get_block_email_html($ip, $reason) {
        $geo_blocker = new ArtistPro_Geo_Blocker();
        $country = $geo_blocker->get_country($ip);
        $country_name = $country ? $country['name'] : 'Unknown';

        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <style>
                body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #6f42c1; color: white; padding: 20px; border-radius: 8px 8px 0 0; }
                .content { background: #f8f9fa; padding: 20px; border-radius: 0 0 8px 8px; }
                .info-table { width: 100%; border-collapse: collapse; margin: 15px 0; }
                .info-table td { padding: 10px; border-bottom: 1px solid #dee2e6; }
                .info-table td:first-child { font-weight: bold; width: 140px; }
                .footer { text-align: center; color: #6c757d; font-size: 12px; margin-top: 20px; }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h2 style="margin:0;">⛔ IP Permanently Blocked</h2>
                </div>
                <div class="content">
                    <p>An IP address has been permanently blocked from accessing your site.</p>

                    <table class="info-table">
                        <tr><td>IP Address</td><td><strong><?php echo esc_html($ip); ?></strong></td></tr>
                        <tr><td>Country</td><td><?php echo esc_html($country_name); ?></td></tr>
                        <tr><td>Reason</td><td><?php echo esc_html($reason); ?></td></tr>
                        <tr><td>Time</td><td><?php echo current_time('mysql'); ?></td></tr>
                    </table>

                    <p>This IP will remain blocked until manually removed from the blacklist.</p>
                </div>
                <div class="footer">
                    <p>This notification was sent by artistpro Security</p>
                </div>
            </div>
        </body>
        </html>
        <?php
        return ob_get_clean();
    }

    /**
     * Get daily summary email HTML
     *
     * @param object $stats Daily statistics
     * @param array $top_blocked Top blocked IPs
     * @return string
     */
    private function get_summary_email_html($stats, $top_blocked) {
        ob_start();
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <style>
                body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background: #28a745; color: white; padding: 20px; border-radius: 8px 8px 0 0; }
                .content { background: #f8f9fa; padding: 20px; border-radius: 0 0 8px 8px; }
                .stats-grid { display: flex; flex-wrap: wrap; gap: 15px; margin: 15px 0; }
                .stat-box { flex: 1; min-width: 120px; background: white; padding: 15px; border-radius: 8px; text-align: center; }
                .stat-number { font-size: 28px; font-weight: bold; color: #007bff; }
                .stat-label { font-size: 12px; color: #6c757d; text-transform: uppercase; }
                .info-table { width: 100%; border-collapse: collapse; margin: 15px 0; }
                .info-table th, .info-table td { padding: 10px; border-bottom: 1px solid #dee2e6; text-align: left; }
                .info-table th { background: #e9ecef; }
                .footer { text-align: center; color: #6c757d; font-size: 12px; margin-top: 20px; }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h2 style="margin:0;">📊 Security Daily Summary</h2>
                    <p style="margin:5px 0 0;opacity:0.9;">Last 24 hours on <?php echo get_bloginfo('name'); ?></p>
                </div>
                <div class="content">
                    <div class="stats-grid">
                        <div class="stat-box">
                            <div class="stat-number"><?php echo number_format($stats->total_attempts); ?></div>
                            <div class="stat-label">Total Attempts</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-number" style="color:#28a745;"><?php echo number_format($stats->successful); ?></div>
                            <div class="stat-label">Successful</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-number" style="color:#dc3545;"><?php echo number_format($stats->failed); ?></div>
                            <div class="stat-label">Failed</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-number" style="color:#6f42c1;"><?php echo number_format($stats->blocked); ?></div>
                            <div class="stat-label">Blocked</div>
                        </div>
                    </div>

                    <p><strong><?php echo number_format($stats->unique_ips); ?></strong> unique IP addresses attempted to log in.</p>

                    <?php if (!empty($top_blocked)): ?>
                    <h3>Top Suspicious IPs</h3>
                    <table class="info-table">
                        <thead>
                            <tr><th>IP Address</th><th>Failed Attempts</th></tr>
                        </thead>
                        <tbody>
                            <?php foreach ($top_blocked as $row): ?>
                            <tr>
                                <td><?php echo esc_html($row->ip_address); ?></td>
                                <td><?php echo number_format($row->count); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <?php endif; ?>
                </div>
                <div class="footer">
                    <p>Generated by artistpro Security at <?php echo current_time('mysql'); ?></p>
                </div>
            </div>
        </body>
        </html>
        <?php
        return ob_get_clean();
    }
}
