<?php
/**
 * Security Cron - Scheduled tasks
 *
 * @package ArtistPro_Security
 * @since 1.0.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_Security_Cron {

    /**
     * Initialize cron hooks
     */
    public function init() {
        // Register cron schedules
        add_filter('cron_schedules', [$this, 'add_cron_schedules']);

        // Register cron hooks
        add_action('artistpro_security_daily_tasks', [$this, 'run_daily_tasks']);
        add_action('artistpro_security_hourly_tasks', [$this, 'run_hourly_tasks']);
        add_action('artistpro_security_sync_blocklists', [$this, 'sync_blocklists']);
        add_action('artistpro_security_download_geoip', [$this, 'download_geoip']);
    }

    /**
     * Add custom cron schedules
     *
     * @param array $schedules
     * @return array
     */
    public function add_cron_schedules($schedules) {
        $schedules['artistpro_twice_daily'] = [
            'interval' => 12 * HOUR_IN_SECONDS,
            'display' => 'Twice Daily'
        ];

        return $schedules;
    }

    /**
     * Schedule cron events
     */
    public static function schedule_events() {
        // Daily tasks - cleanup, reports
        if (!wp_next_scheduled('artistpro_security_daily_tasks')) {
            wp_schedule_event(time(), 'daily', 'artistpro_security_daily_tasks');
        }

        // Hourly tasks - expire lockouts
        if (!wp_next_scheduled('artistpro_security_hourly_tasks')) {
            wp_schedule_event(time(), 'hourly', 'artistpro_security_hourly_tasks');
        }

        // Blocklist sync - twice daily
        if (!wp_next_scheduled('artistpro_security_sync_blocklists')) {
            wp_schedule_event(time() + 300, 'artistpro_twice_daily', 'artistpro_security_sync_blocklists');
        }
    }

    /**
     * Unschedule all cron events
     */
    public static function unschedule_events() {
        wp_clear_scheduled_hook('artistpro_security_daily_tasks');
        wp_clear_scheduled_hook('artistpro_security_hourly_tasks');
        wp_clear_scheduled_hook('artistpro_security_sync_blocklists');
        wp_clear_scheduled_hook('artistpro_security_download_geoip');
    }

    /**
     * Run daily tasks
     */
    public function run_daily_tasks() {
        // Clean old logs
        $this->cleanup_old_logs();

        // Clean expired lockouts
        $this->cleanup_expired_lockouts();

        // Clean expired IP list entries
        $this->cleanup_expired_ip_entries();

        // Send daily summary if enabled
        $this->send_daily_summary();

        // Update GeoIP database monthly
        $this->check_geoip_update();
    }

    /**
     * Run hourly tasks
     */
    public function run_hourly_tasks() {
        // Remove expired lockouts
        $this->cleanup_expired_lockouts();
    }

    /**
     * Sync blocklists
     */
    public function sync_blocklists() {
        $blocklist_sync = new ArtistPro_Blocklist_Sync();
        $results = $blocklist_sync->sync_all();

        // Log results
        $total = 0;
        foreach ($results as $source => $result) {
            if ($result['success']) {
                $total += $result['count'];
            }
        }

        error_log("ArtistPro Security: Blocklist sync complete - {$total} IPs loaded");
    }

    /**
     * Download GeoIP database
     */
    public function download_geoip() {
        $geo_blocker = new ArtistPro_Geo_Blocker();
        $geo_blocker->download_database();
    }

    /**
     * Cleanup old log entries
     */
    private function cleanup_old_logs() {
        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_logs';

        $retention_days = get_option('artistpro_security_log_retention', 30);

        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM {$table} WHERE created_at < DATE_SUB(NOW(), INTERVAL %d DAY)",
            $retention_days
        ));

        if ($deleted > 0) {
            error_log("ArtistPro Security: Cleaned up {$deleted} old log entries");
        }
    }

    /**
     * Cleanup expired lockouts
     */
    private function cleanup_expired_lockouts() {
        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_lockouts';

        $deleted = $wpdb->query(
            "DELETE FROM {$table}
             WHERE locked_until IS NOT NULL
               AND locked_until < NOW()
               AND lockout_type != 'permanent'"
        );

        if ($deleted > 0) {
            error_log("ArtistPro Security: Removed {$deleted} expired lockouts");
        }
    }

    /**
     * Cleanup expired IP list entries
     */
    private function cleanup_expired_ip_entries() {
        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_ip_lists';

        $deleted = $wpdb->query(
            "DELETE FROM {$table}
             WHERE expires_at IS NOT NULL AND expires_at < NOW()"
        );

        if ($deleted > 0) {
            error_log("ArtistPro Security: Removed {$deleted} expired IP entries");
        }
    }

    /**
     * Send daily summary email
     */
    private function send_daily_summary() {
        if (!get_option('artistpro_security_daily_summary', false)) {
            return;
        }

        global $wpdb;
        $logs_table = $wpdb->prefix . 'artistpro_security_logs';

        // Get stats for last 24 hours
        $stats = $wpdb->get_row("
            SELECT
                COUNT(*) as total_attempts,
                SUM(CASE WHEN result = 'success' THEN 1 ELSE 0 END) as successful,
                SUM(CASE WHEN result = 'failed' THEN 1 ELSE 0 END) as failed,
                SUM(CASE WHEN result = 'blocked' THEN 1 ELSE 0 END) as blocked,
                SUM(CASE WHEN result = 'lockout' THEN 1 ELSE 0 END) as lockouts,
                COUNT(DISTINCT ip_address) as unique_ips
            FROM {$logs_table}
            WHERE created_at > DATE_SUB(NOW(), INTERVAL 24 HOUR)
        ");

        // Skip if no activity
        if ($stats->total_attempts == 0) {
            return;
        }

        // Get top blocked IPs
        $top_blocked = $wpdb->get_results("
            SELECT ip_address, COUNT(*) as count
            FROM {$logs_table}
            WHERE created_at > DATE_SUB(NOW(), INTERVAL 24 HOUR)
              AND result IN ('failed', 'blocked', 'lockout')
            GROUP BY ip_address
            ORDER BY count DESC
            LIMIT 5
        ");

        // Build email
        $notifier = new ArtistPro_Security_Notifier();
        $notifier->send_daily_summary($stats, $top_blocked);
    }

    /**
     * Check if GeoIP database needs update
     */
    private function check_geoip_update() {
        $last_update = get_option('artistpro_security_geoip_updated', 0);

        // Update monthly
        if ($last_update < strtotime('-30 days')) {
            $geo_blocker = new ArtistPro_Geo_Blocker();
            $geo_blocker->schedule_database_download();
        }
    }
}
