<?php
/**
 * Security Core - Main throttling and blocking logic
 *
 * @package ArtistPro_Security
 * @since 1.0.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_Security_Core {

    /**
     * Track if we've already shown an error
     */
    private static $error_shown = false;

    /**
     * Initialize the class and register hooks
     */
    public static function init() {
        // Early authentication check (priority 0 - runs first)
        add_filter('authenticate', array(__CLASS__, 'check_before_auth'), 0, 3);

        // Track failed logins
        add_action('wp_login_failed', array(__CLASS__, 'on_login_failed'), 10, 2);

        // Track successful logins
        add_action('wp_login', array(__CLASS__, 'on_login_success'), 10, 2);

        // Final lockout check (priority 99999 - runs last before user object is returned)
        add_filter('wp_authenticate_user', array(__CLASS__, 'check_lockout_before_user'), 99999, 2);

        // Customize error messages
        add_filter('login_errors', array(__CLASS__, 'customize_error_message'));

        // XMLRPC authentication
        add_filter('xmlrpc_login_error', array(__CLASS__, 'handle_xmlrpc_error'), 10, 2);

        // REST API authentication
        add_filter('rest_authentication_errors', array(__CLASS__, 'check_rest_auth'), 99);
    }

    /**
     * Check before authentication starts
     * Blocks blacklisted IPs and blocked countries before any password check
     *
     * @param WP_User|WP_Error|null $user User object or error
     * @param string $username Username
     * @param string $password Password
     * @return WP_User|WP_Error|null
     */
    public static function check_before_auth($user, $username, $password) {
        // Skip if already an error or no username provided
        if (is_wp_error($user) || empty($username)) {
            return $user;
        }

        $ip = ArtistPro_IP_Manager::get_client_ip();

        // Check whitelist first - whitelisted IPs bypass all checks
        if (ArtistPro_IP_Manager::is_whitelisted($ip)) {
            return $user;
        }

        // Check blacklist
        if (ArtistPro_IP_Manager::is_blacklisted($ip)) {
            ArtistPro_Login_Logger::log_attempt(array(
                'ip_address' => $ip,
                'username' => $username,
                'login_type' => 'wp_login',
                'result' => 'blocked',
                'failure_reason' => 'IP blacklisted',
            ));

            return new WP_Error(
                'artistpro_security_blocked',
                __('Access denied.', 'artistpro-security')
            );
        }

        // Check country blocking
        if (class_exists('ArtistPro_Geo_Blocker') && ArtistPro_Geo_Blocker::is_blocked($ip)) {
            ArtistPro_Login_Logger::log_attempt(array(
                'ip_address' => $ip,
                'username' => $username,
                'login_type' => 'wp_login',
                'result' => 'blocked',
                'failure_reason' => 'Country blocked',
            ));

            return new WP_Error(
                'artistpro_security_country_blocked',
                __('Access denied.', 'artistpro-security')
            );
        }

        // Check if already locked out
        if (self::is_locked_out($ip)) {
            $remaining = self::get_lockout_remaining($ip);

            ArtistPro_Login_Logger::log_attempt(array(
                'ip_address' => $ip,
                'username' => $username,
                'login_type' => 'wp_login',
                'result' => 'lockout',
                'failure_reason' => 'IP locked out',
            ));

            return new WP_Error(
                'artistpro_security_locked_out',
                sprintf(
                    __('Too many failed login attempts. Please try again in %s.', 'artistpro-security'),
                    self::format_duration($remaining)
                )
            );
        }

        return $user;
    }

    /**
     * Check lockout status before returning user object
     *
     * @param WP_User|WP_Error $user User object or error
     * @param string $password Password
     * @return WP_User|WP_Error
     */
    public static function check_lockout_before_user($user, $password) {
        if (is_wp_error($user)) {
            return $user;
        }

        $ip = ArtistPro_IP_Manager::get_client_ip();

        // Whitelisted IPs bypass lockout
        if (ArtistPro_IP_Manager::is_whitelisted($ip)) {
            return $user;
        }

        if (self::is_locked_out($ip)) {
            $remaining = self::get_lockout_remaining($ip);

            return new WP_Error(
                'artistpro_security_locked_out',
                sprintf(
                    __('Too many failed login attempts. Please try again in %s.', 'artistpro-security'),
                    self::format_duration($remaining)
                )
            );
        }

        return $user;
    }

    /**
     * Handle failed login attempt
     *
     * @param string $username Username
     * @param WP_Error $error Error object
     */
    public static function on_login_failed($username, $error = null) {
        $ip = ArtistPro_IP_Manager::get_client_ip();

        // Skip if whitelisted
        if (ArtistPro_IP_Manager::is_whitelisted($ip)) {
            return;
        }

        // Determine failure reason
        $reason = 'Invalid credentials';
        if ($error instanceof WP_Error) {
            $error_code = $error->get_error_code();
            if ($error_code === 'invalid_username') {
                $reason = 'Invalid username';
            } elseif ($error_code === 'incorrect_password') {
                $reason = 'Incorrect password';
            } elseif ($error_code === 'invalid_email') {
                $reason = 'Invalid email';
            }
        }

        // Log the attempt
        ArtistPro_Login_Logger::log_attempt(array(
            'ip_address' => $ip,
            'username' => $username,
            'login_type' => 'wp_login',
            'result' => 'failed',
            'failure_reason' => $reason,
        ));

        // Check if we should trigger a lockout
        self::maybe_trigger_lockout($ip, $username);
    }

    /**
     * Handle successful login
     *
     * @param string $username Username
     * @param WP_User $user User object
     */
    public static function on_login_success($username, $user) {
        $ip = ArtistPro_IP_Manager::get_client_ip();

        // Log successful login
        ArtistPro_Login_Logger::log_attempt(array(
            'ip_address' => $ip,
            'username' => $username,
            'email' => $user->user_email,
            'login_type' => 'wp_login',
            'result' => 'success',
        ));

        // Clear any existing lockout for this IP
        self::clear_lockout($ip);
    }

    /**
     * Check if IP is currently locked out
     *
     * @param string $ip IP address
     * @return bool
     */
    public static function is_locked_out($ip) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOCKOUTS;

        $lockout = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE ip_address = %s AND locked_until > NOW()",
            $ip
        ));

        return !empty($lockout);
    }

    /**
     * Get remaining lockout time in seconds
     *
     * @param string $ip IP address
     * @return int Seconds remaining
     */
    public static function get_lockout_remaining($ip) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOCKOUTS;

        $locked_until = $wpdb->get_var($wpdb->prepare(
            "SELECT locked_until FROM $table WHERE ip_address = %s AND locked_until > NOW()",
            $ip
        ));

        if ($locked_until) {
            return max(0, strtotime($locked_until) - time());
        }

        return 0;
    }

    /**
     * Get remaining login attempts for an IP
     *
     * @param string $ip IP address
     * @return int
     */
    public static function get_remaining_attempts($ip) {
        $allowed = self::get_allowed_retries();
        $valid_duration = get_option('artistpro_security_valid_duration', 43200);

        $attempts = ArtistPro_Login_Logger::get_attempts_by_ip($ip, $valid_duration);
        $remaining = $allowed - ($attempts % $allowed);

        return max(0, $remaining);
    }

    /**
     * Check if we should trigger a lockout
     *
     * @param string $ip IP address
     * @param string $username Username
     */
    private static function maybe_trigger_lockout($ip, $username) {
        $allowed_retries = self::get_allowed_retries();
        $lockouts_before_long = get_option('artistpro_security_lockouts_before_long', 4);
        $valid_duration = get_option('artistpro_security_valid_duration', 43200);

        // Count failures in the valid window
        $failures = ArtistPro_Login_Logger::get_attempts_by_ip($ip, $valid_duration);

        // Check if we've hit a lockout threshold
        if ($failures > 0 && $failures % $allowed_retries === 0) {
            // Determine lockout type
            $lockout_count = floor($failures / $allowed_retries);

            if ($lockout_count >= $lockouts_before_long) {
                // Long lockout
                self::trigger_lockout($ip, $username, 'long');
            } else {
                // Short lockout
                self::trigger_lockout($ip, $username, 'short');
            }
        }

        // Check auto-blacklist threshold
        $auto_threshold = get_option('artistpro_security_auto_blacklist_threshold', 10);
        if ($auto_threshold > 0 && $failures >= $auto_threshold) {
            // Auto-add to blacklist
            ArtistPro_IP_Manager::add_to_blacklist(
                $ip,
                sprintf('Auto-blocked after %d failed attempts', $failures),
                'auto_blocked',
                date('Y-m-d H:i:s', strtotime('+7 days')) // Expire in 7 days
            );
        }
    }

    /**
     * Trigger a lockout for an IP
     *
     * @param string $ip IP address
     * @param string $username Username that triggered lockout
     * @param string $type 'short' or 'long'
     */
    public static function trigger_lockout($ip, $username, $type = 'short') {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOCKOUTS;

        // Calculate lockout duration
        if ($type === 'long') {
            $duration = get_option('artistpro_security_long_lockout_duration', 24) * 3600; // hours to seconds
        } else {
            $duration = get_option('artistpro_security_lockout_duration', 20) * 60; // minutes to seconds
        }

        $locked_until = date('Y-m-d H:i:s', time() + $duration);

        // Get failure count
        $valid_duration = get_option('artistpro_security_valid_duration', 43200);
        $failures = ArtistPro_Login_Logger::get_attempts_by_ip($ip, $valid_duration);

        // Insert or update lockout record
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table WHERE ip_address = %s",
            $ip
        ));

        if ($existing) {
            $wpdb->update(
                $table,
                array(
                    'lockout_type' => $type,
                    'attempts_count' => $failures,
                    'locked_until' => $locked_until,
                    'username' => $username,
                    'reason' => sprintf('%d failed attempts', $failures),
                ),
                array('id' => $existing),
                array('%s', '%d', '%s', '%s', '%s'),
                array('%d')
            );
        } else {
            $wpdb->insert(
                $table,
                array(
                    'ip_address' => $ip,
                    'username' => $username,
                    'lockout_type' => $type,
                    'attempts_count' => $failures,
                    'locked_until' => $locked_until,
                    'reason' => sprintf('%d failed attempts', $failures),
                ),
                array('%s', '%s', '%s', '%d', '%s', '%s')
            );
        }

        // Send notification if configured
        $notify_after = get_option('artistpro_security_notify_after_lockouts', 3);
        $lockout_count = floor($failures / self::get_allowed_retries());

        if ($lockout_count >= $notify_after && class_exists('ArtistPro_Security_Notifier')) {
            ArtistPro_Security_Notifier::send_lockout_notification($ip, $username, $failures);
        }

        error_log(sprintf(
            'artistpro Security: Locked out IP %s for %s (%d failed attempts, user: %s)',
            $ip,
            self::format_duration($duration),
            $failures,
            $username
        ));
    }

    /**
     * Clear lockout for an IP
     *
     * @param string $ip IP address
     * @return bool
     */
    public static function clear_lockout($ip) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOCKOUTS;

        return $wpdb->delete(
            $table,
            array('ip_address' => $ip),
            array('%s')
        ) !== false;
    }

    /**
     * Clear all lockouts
     *
     * @return int Number of cleared lockouts
     */
    public static function clear_all_lockouts() {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOCKOUTS;

        return $wpdb->query("TRUNCATE TABLE $table");
    }

    /**
     * Get all active lockouts
     *
     * @return array
     */
    public static function get_active_lockouts() {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOCKOUTS;

        return $wpdb->get_results(
            "SELECT * FROM $table WHERE locked_until > NOW() ORDER BY created_at DESC"
        );
    }

    /**
     * Clean up expired lockouts
     *
     * @return int Number of deleted rows
     */
    public static function cleanup_expired_lockouts() {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOCKOUTS;

        return $wpdb->query("DELETE FROM $table WHERE locked_until <= NOW()");
    }

    /**
     * Customize login error messages
     *
     * @param string $error Error message
     * @return string
     */
    public static function customize_error_message($error) {
        $ip = ArtistPro_IP_Manager::get_client_ip();

        if (self::is_locked_out($ip)) {
            $remaining = self::get_lockout_remaining($ip);
            return sprintf(
                '<strong>%s</strong> %s',
                __('Error:', 'artistpro-security'),
                sprintf(
                    __('Too many failed login attempts. Please try again in %s.', 'artistpro-security'),
                    self::format_duration($remaining)
                )
            );
        }

        // Add remaining attempts notice
        $remaining_attempts = self::get_remaining_attempts($ip);
        if ($remaining_attempts <= 3 && $remaining_attempts > 0) {
            $error .= '<br><br>' . sprintf(
                __('<strong>Warning:</strong> %d login attempts remaining before lockout.', 'artistpro-security'),
                $remaining_attempts
            );
        }

        return $error;
    }

    /**
     * Handle XMLRPC login errors
     *
     * @param IXR_Error $error Error object
     * @param WP_Error $user WP_Error from failed login
     * @return IXR_Error
     */
    public static function handle_xmlrpc_error($error, $user) {
        $ip = ArtistPro_IP_Manager::get_client_ip();

        // Log XMLRPC attempt
        ArtistPro_Login_Logger::log_attempt(array(
            'ip_address' => $ip,
            'login_type' => 'xmlrpc',
            'result' => 'failed',
            'failure_reason' => 'XMLRPC authentication failed',
        ));

        // Check lockout
        self::maybe_trigger_lockout($ip, 'xmlrpc');

        return $error;
    }

    /**
     * Check REST API authentication
     *
     * @param WP_Error|null|bool $result Authentication result
     * @return WP_Error|null|bool
     */
    public static function check_rest_auth($result) {
        if (is_wp_error($result)) {
            return $result;
        }

        $ip = ArtistPro_IP_Manager::get_client_ip();

        // Check blacklist for API requests
        if (ArtistPro_IP_Manager::is_blacklisted($ip)) {
            return new WP_Error(
                'rest_forbidden',
                __('Access denied.', 'artistpro-security'),
                array('status' => 403)
            );
        }

        // Check lockout
        if (self::is_locked_out($ip)) {
            return new WP_Error(
                'rest_forbidden',
                __('Too many requests. Please try again later.', 'artistpro-security'),
                array('status' => 429)
            );
        }

        return $result;
    }

    /**
     * Get allowed retries setting
     *
     * @return int
     */
    public static function get_allowed_retries() {
        return (int) get_option('artistpro_security_allowed_retries', 4);
    }

    /**
     * Format duration in human readable format
     *
     * @param int $seconds Seconds
     * @return string
     */
    public static function format_duration($seconds) {
        if ($seconds < 60) {
            return sprintf(_n('%d second', '%d seconds', $seconds, 'artistpro-security'), $seconds);
        }

        $minutes = floor($seconds / 60);
        if ($minutes < 60) {
            return sprintf(_n('%d minute', '%d minutes', $minutes, 'artistpro-security'), $minutes);
        }

        $hours = floor($minutes / 60);
        $remaining_minutes = $minutes % 60;

        if ($remaining_minutes > 0) {
            return sprintf(
                __('%d hours %d minutes', 'artistpro-security'),
                $hours,
                $remaining_minutes
            );
        }

        return sprintf(_n('%d hour', '%d hours', $hours, 'artistpro-security'), $hours);
    }
}
