<?php
/**
 * Security AJAX - Admin AJAX handlers
 *
 * @package ArtistPro_Security
 * @since 1.0.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_Security_Ajax {

    /**
     * Initialize AJAX hooks
     */
    public function init() {
        // IP management
        add_action('wp_ajax_artistpro_security_add_ip', [$this, 'add_ip']);
        add_action('wp_ajax_artistpro_security_remove_ip', [$this, 'remove_ip']);
        add_action('wp_ajax_artistpro_security_unlock_ip', [$this, 'unlock_ip']);

        // Country blocking
        add_action('wp_ajax_artistpro_security_block_country', [$this, 'block_country']);
        add_action('wp_ajax_artistpro_security_unblock_country', [$this, 'unblock_country']);

        // Blocklist management
        add_action('wp_ajax_artistpro_security_sync_blocklist', [$this, 'sync_blocklist']);
        add_action('wp_ajax_artistpro_security_toggle_blocklist', [$this, 'toggle_blocklist']);

        // Log management
        add_action('wp_ajax_artistpro_security_clear_logs', [$this, 'clear_logs']);
        add_action('wp_ajax_artistpro_security_export_logs', [$this, 'export_logs']);

        // Settings
        add_action('wp_ajax_artistpro_security_save_settings', [$this, 'save_settings']);
    }

    /**
     * Verify AJAX request
     *
     * @return bool
     */
    private function verify_request() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Permission denied']);
            return false;
        }

        if (!check_ajax_referer('artistpro_security_nonce', 'nonce', false)) {
            wp_send_json_error(['message' => 'Invalid nonce']);
            return false;
        }

        return true;
    }

    /**
     * Add IP to whitelist or blacklist
     */
    public function add_ip() {
        if (!$this->verify_request()) return;

        $ip = sanitize_text_field($_POST['ip'] ?? '');
        $list_type = sanitize_text_field($_POST['list_type'] ?? 'blacklist');
        $reason = sanitize_text_field($_POST['reason'] ?? '');
        $expires = sanitize_text_field($_POST['expires'] ?? '');

        if (empty($ip)) {
            wp_send_json_error(['message' => 'IP address required']);
            return;
        }

        $ip_manager = new ArtistPro_IP_Manager();
        $result = $ip_manager->add_to_list($ip, $list_type, 'manual', $reason, $expires ?: null);

        if ($result) {
            wp_send_json_success(['message' => 'IP added successfully']);
        } else {
            wp_send_json_error(['message' => 'Failed to add IP']);
        }
    }

    /**
     * Remove IP from list
     */
    public function remove_ip() {
        if (!$this->verify_request()) return;

        $ip = sanitize_text_field($_POST['ip'] ?? '');

        if (empty($ip)) {
            wp_send_json_error(['message' => 'IP address required']);
            return;
        }

        $ip_manager = new ArtistPro_IP_Manager();
        $result = $ip_manager->remove_from_list($ip);

        if ($result) {
            wp_send_json_success(['message' => 'IP removed successfully']);
        } else {
            wp_send_json_error(['message' => 'Failed to remove IP']);
        }
    }

    /**
     * Unlock locked IP
     */
    public function unlock_ip() {
        if (!$this->verify_request()) return;

        $ip = sanitize_text_field($_POST['ip'] ?? '');

        if (empty($ip)) {
            wp_send_json_error(['message' => 'IP address required']);
            return;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_lockouts';

        $result = $wpdb->delete($table, ['ip_address' => $ip], ['%s']);

        if ($result) {
            wp_send_json_success(['message' => 'IP unlocked successfully']);
        } else {
            wp_send_json_error(['message' => 'IP not found in lockouts']);
        }
    }

    /**
     * Block country
     */
    public function block_country() {
        if (!$this->verify_request()) return;

        $country_code = sanitize_text_field($_POST['country_code'] ?? '');
        $block_type = sanitize_text_field($_POST['block_type'] ?? 'all');

        if (empty($country_code) || strlen($country_code) !== 2) {
            wp_send_json_error(['message' => 'Invalid country code']);
            return;
        }

        $geo_blocker = new ArtistPro_Geo_Blocker();
        $result = $geo_blocker->block_country($country_code, $block_type);

        if ($result) {
            wp_send_json_success(['message' => 'Country blocked successfully']);
        } else {
            wp_send_json_error(['message' => 'Failed to block country']);
        }
    }

    /**
     * Unblock country
     */
    public function unblock_country() {
        if (!$this->verify_request()) return;

        $country_code = sanitize_text_field($_POST['country_code'] ?? '');

        if (empty($country_code)) {
            wp_send_json_error(['message' => 'Country code required']);
            return;
        }

        $geo_blocker = new ArtistPro_Geo_Blocker();
        $result = $geo_blocker->unblock_country($country_code);

        if ($result) {
            wp_send_json_success(['message' => 'Country unblocked']);
        } else {
            wp_send_json_error(['message' => 'Failed to unblock country']);
        }
    }

    /**
     * Sync blocklist
     */
    public function sync_blocklist() {
        if (!$this->verify_request()) return;

        $source = sanitize_text_field($_POST['source'] ?? '');

        $blocklist_sync = new ArtistPro_Blocklist_Sync();

        if ($source) {
            $result = $blocklist_sync->sync_source($source);
            wp_send_json_success($result);
        } else {
            $results = $blocklist_sync->sync_all();
            wp_send_json_success($results);
        }
    }

    /**
     * Toggle blocklist source
     */
    public function toggle_blocklist() {
        if (!$this->verify_request()) return;

        $source = sanitize_text_field($_POST['source'] ?? '');
        $enabled = filter_var($_POST['enabled'] ?? false, FILTER_VALIDATE_BOOLEAN);

        if (empty($source)) {
            wp_send_json_error(['message' => 'Source required']);
            return;
        }

        $blocklist_sync = new ArtistPro_Blocklist_Sync();

        if ($enabled) {
            $blocklist_sync->enable_source($source);
            // Sync immediately
            $result = $blocklist_sync->sync_source($source);
            wp_send_json_success(['message' => 'Blocklist enabled and synced', 'sync_result' => $result]);
        } else {
            $blocklist_sync->disable_source($source);
            wp_send_json_success(['message' => 'Blocklist disabled']);
        }
    }

    /**
     * Clear logs
     */
    public function clear_logs() {
        if (!$this->verify_request()) return;

        $days = intval($_POST['days'] ?? 0);

        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_logs';

        if ($days > 0) {
            $deleted = $wpdb->query($wpdb->prepare(
                "DELETE FROM {$table} WHERE created_at < DATE_SUB(NOW(), INTERVAL %d DAY)",
                $days
            ));
        } else {
            $deleted = $wpdb->query("TRUNCATE TABLE {$table}");
        }

        wp_send_json_success(['message' => "Logs cleared", 'deleted' => $deleted]);
    }

    /**
     * Export logs to CSV
     */
    public function export_logs() {
        if (!$this->verify_request()) return;

        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_logs';

        $logs = $wpdb->get_results("SELECT * FROM {$table} ORDER BY created_at DESC LIMIT 10000", ARRAY_A);

        if (empty($logs)) {
            wp_send_json_error(['message' => 'No logs to export']);
            return;
        }

        // Generate CSV
        $csv = fopen('php://temp', 'r+');

        // Header row
        fputcsv($csv, array_keys($logs[0]));

        // Data rows
        foreach ($logs as $row) {
            fputcsv($csv, $row);
        }

        rewind($csv);
        $content = stream_get_contents($csv);
        fclose($csv);

        wp_send_json_success([
            'filename' => 'security-logs-' . date('Y-m-d') . '.csv',
            'content' => base64_encode($content)
        ]);
    }

    /**
     * Save settings
     */
    public function save_settings() {
        if (!$this->verify_request()) return;

        $settings = [
            'allowed_retries' => intval($_POST['allowed_retries'] ?? 4),
            'short_lockout_duration' => intval($_POST['short_lockout_duration'] ?? 20),
            'long_lockout_duration' => intval($_POST['long_lockout_duration'] ?? 1440),
            'lockouts_before_long' => intval($_POST['lockouts_before_long'] ?? 4),
            'auto_blacklist_threshold' => intval($_POST['auto_blacklist_threshold'] ?? 10),
            'log_retention' => intval($_POST['log_retention'] ?? 30),
            'notify_lockouts' => filter_var($_POST['notify_lockouts'] ?? true, FILTER_VALIDATE_BOOLEAN),
            'notify_blocks' => filter_var($_POST['notify_blocks'] ?? true, FILTER_VALIDATE_BOOLEAN),
            'daily_summary' => filter_var($_POST['daily_summary'] ?? false, FILTER_VALIDATE_BOOLEAN),
            'notify_email' => sanitize_email($_POST['notify_email'] ?? ''),
            'maxmind_key' => sanitize_text_field($_POST['maxmind_key'] ?? '')
        ];

        foreach ($settings as $key => $value) {
            update_option('artistpro_security_' . $key, $value);
        }

        wp_send_json_success(['message' => 'Settings saved']);
    }
}
