<?php
/**
 * reCAPTCHA Integration
 * Google reCAPTCHA v3 (invisible) and v2 (checkbox) support
 *
 * @package ArtistPro_Security
 * @since 1.1.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_reCAPTCHA {

    /**
     * Initialize reCAPTCHA
     */
    public static function init() {
        if (!self::is_enabled()) {
            return;
        }

        // Add reCAPTCHA to login form
        add_action('login_enqueue_scripts', array(__CLASS__, 'enqueue_scripts'));
        add_action('login_form', array(__CLASS__, 'add_recaptcha_field'));
        
        // Validate reCAPTCHA
        add_filter('authenticate', array(__CLASS__, 'validate_recaptcha'), 30, 3);
        
        // Optional: Add to registration
        if (get_option('artistpro_security_recaptcha_on_registration', false)) {
            add_action('register_form', array(__CLASS__, 'add_recaptcha_field'));
            add_filter('registration_errors', array(__CLASS__, 'validate_recaptcha_registration'), 10, 3);
        }
        
        // Optional: Add to password reset
        if (get_option('artistpro_security_recaptcha_on_reset', false)) {
            add_action('lostpassword_form', array(__CLASS__, 'add_recaptcha_field'));
        }
        
        // Settings
        add_action('admin_init', array(__CLASS__, 'register_settings'));
    }

    /**
     * Check if reCAPTCHA is enabled
     *
     * @return bool
     */
    public static function is_enabled() {
        return (bool) get_option('artistpro_security_recaptcha_enabled', false);
    }

    /**
     * Get reCAPTCHA version
     *
     * @return string 'v2' or 'v3'
     */
    private static function get_version() {
        return get_option('artistpro_security_recaptcha_version', 'v3');
    }

    /**
     * Get site key
     *
     * @return string
     */
    private static function get_site_key() {
        return get_option('artistpro_security_recaptcha_site_key', '');
    }

    /**
     * Get secret key
     *
     * @return string
     */
    private static function get_secret_key() {
        return get_option('artistpro_security_recaptcha_secret_key', '');
    }

    /**
     * Enqueue reCAPTCHA scripts
     */
    public static function enqueue_scripts() {
        $version = self::get_version();
        $site_key = self::get_site_key();
        
        if (empty($site_key)) {
            return;
        }

        if ($version === 'v3') {
            wp_enqueue_script(
                'google-recaptcha-v3',
                "https://www.google.com/recaptcha/api.js?render={$site_key}",
                array(),
                null,
                true
            );
        } else {
            wp_enqueue_script(
                'google-recaptcha-v2',
                'https://www.google.com/recaptcha/api.js',
                array(),
                null,
                true
            );
        }
    }

    /**
     * Add reCAPTCHA field to form
     */
    public static function add_recaptcha_field() {
        $version = self::get_version();
        $site_key = self::get_site_key();
        
        if (empty($site_key)) {
            return;
        }

        if ($version === 'v3') {
            ?>
            <input type="hidden" name="g-recaptcha-response" id="g-recaptcha-response">
            <script>
            grecaptcha.ready(function() {
                grecaptcha.execute('<?php echo esc_js($site_key); ?>', {action: 'login'})
                    .then(function(token) {
                        document.getElementById('g-recaptcha-response').value = token;
                    });
            });
            </script>
            <?php
        } else {
            ?>
            <p>
                <div class="g-recaptcha" data-sitekey="<?php echo esc_attr($site_key); ?>"></div>
            </p>
            <?php
        }
    }

    /**
     * Validate reCAPTCHA response
     *
     * @param WP_User|WP_Error|null $user User object or error
     * @param string $username Username
     * @param string $password Password
     * @return WP_User|WP_Error
     */
    public static function validate_recaptcha($user, $username, $password) {
        // Skip if already an error
        if (is_wp_error($user)) {
            return $user;
        }

        // Skip validation for whitelisted IPs
        $ip = ArtistPro_IP_Manager::get_client_ip();
        if (ArtistPro_IP_Manager::is_whitelisted($ip)) {
            return $user;
        }

        // Check if we should show reCAPTCHA based on failed attempts
        $threshold = (int) get_option('artistpro_security_recaptcha_threshold', 2);
        if ($threshold > 0) {
            $failed_attempts = ArtistPro_Login_Logger::get_attempts_by_ip($ip, 3600);
            if ($failed_attempts < $threshold) {
                return $user; // Not enough failures yet
            }
        }

        $response = isset($_POST['g-recaptcha-response']) ? $_POST['g-recaptcha-response'] : '';
        
        if (empty($response)) {
            return new WP_Error(
                'recaptcha_missing',
                __('Please complete the reCAPTCHA verification.', 'artistpro-security')
            );
        }

        $verified = self::verify_response($response, $ip);
        
        if (!$verified) {
            return new WP_Error(
                'recaptcha_failed',
                __('reCAPTCHA verification failed. Please try again.', 'artistpro-security')
            );
        }

        return $user;
    }

    /**
     * Validate reCAPTCHA for registration
     *
     * @param WP_Error $errors Error object
     * @param string $sanitized_user_login Sanitized username
     * @param string $user_email User email
     * @return WP_Error
     */
    public static function validate_recaptcha_registration($errors, $sanitized_user_login, $user_email) {
        $response = isset($_POST['g-recaptcha-response']) ? $_POST['g-recaptcha-response'] : '';
        
        if (empty($response)) {
            $errors->add('recaptcha_missing', __('Please complete the reCAPTCHA verification.', 'artistpro-security'));
            return $errors;
        }

        $ip = ArtistPro_IP_Manager::get_client_ip();
        $verified = self::verify_response($response, $ip);
        
        if (!$verified) {
            $errors->add('recaptcha_failed', __('reCAPTCHA verification failed.', 'artistpro-security'));
        }

        return $errors;
    }

    /**
     * Verify reCAPTCHA response with Google
     *
     * @param string $response reCAPTCHA response token
     * @param string $ip User IP address
     * @return bool
     */
    private static function verify_response($response, $ip) {
        $secret_key = self::get_secret_key();
        
        if (empty($secret_key)) {
            return false;
        }

        $verify_url = 'https://www.google.com/recaptcha/api/siteverify';
        
        $response = wp_remote_post($verify_url, array(
            'body' => array(
                'secret' => $secret_key,
                'response' => $response,
                'remoteip' => $ip,
            ),
        ));

        if (is_wp_error($response)) {
            error_log('reCAPTCHA verification error: ' . $response->get_error_message());
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);

        if (!isset($result['success'])) {
            return false;
        }

        // For v3, check score
        if (self::get_version() === 'v3' && isset($result['score'])) {
            $min_score = (float) get_option('artistpro_security_recaptcha_v3_score', 0.5);
            return $result['success'] && $result['score'] >= $min_score;
        }

        return $result['success'];
    }

    /**
     * Register settings
     */
    public static function register_settings() {
        register_setting('artistpro_security_recaptcha', 'artistpro_security_recaptcha_enabled');
        register_setting('artistpro_security_recaptcha', 'artistpro_security_recaptcha_version');
        register_setting('artistpro_security_recaptcha', 'artistpro_security_recaptcha_site_key');
        register_setting('artistpro_security_recaptcha', 'artistpro_security_recaptcha_secret_key');
        register_setting('artistpro_security_recaptcha', 'artistpro_security_recaptcha_threshold');
        register_setting('artistpro_security_recaptcha', 'artistpro_security_recaptcha_v3_score');
        register_setting('artistpro_security_recaptcha', 'artistpro_security_recaptcha_on_registration');
        register_setting('artistpro_security_recaptcha', 'artistpro_security_recaptcha_on_reset');
    }
}
