<?php
/**
 * Portal Integration - Hook into artistpro customer portal auth
 *
 * @package ArtistPro_Security
 * @since 1.0.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_Portal_Integration {

    /**
     * @var ArtistPro_Security_Core
     */
    private $security_core;

    /**
     * @var ArtistPro_Login_Logger
     */
    private $logger;

    /**
     * @var ArtistPro_IP_Manager
     */
    private $ip_manager;

    /**
     * Constructor
     */
    public function __construct() {
        $this->security_core = new ArtistPro_Security_Core();
        $this->logger = new ArtistPro_Login_Logger();
        $this->ip_manager = new ArtistPro_IP_Manager();
    }

    /**
     * Initialize hooks
     */
    public function init() {
        // Check if portal plugin is active
        if (!class_exists('ArtistPro_Customer_Portal')) {
            return;
        }

        // Hook into portal auth events
        add_action('artistpro_security_before_magic_link', [$this, 'before_magic_link'], 10, 1);
        add_action('artistpro_security_before_sms', [$this, 'before_sms'], 10, 1);
        add_action('artistpro_security_portal_login', [$this, 'on_portal_login'], 10, 3);

        // Add early check filter
        add_filter('artistpro_portal_auth_check', [$this, 'check_auth_allowed'], 10, 2);
    }

    /**
     * Check if authentication is allowed for this IP
     *
     * @param bool $allowed Current allowed status
     * @param string $method Auth method (email, sms, passkey)
     * @return bool|WP_Error
     */
    public function check_auth_allowed($allowed, $method) {
        if (!$allowed) {
            return $allowed;
        }

        $ip = $this->ip_manager->get_client_ip();

        // Check if IP is blocked
        $check = $this->security_core->check_ip_before_auth($ip);

        if ($check['blocked']) {
            $this->logger->log([
                'ip_address' => $ip,
                'username' => '',
                'login_type' => 'portal_' . $method,
                'result' => 'blocked',
                'failure_reason' => $check['reason']
            ]);

            return new WP_Error('blocked', $check['message']);
        }

        // Check if IP is locked out
        $lockout = $this->security_core->get_active_lockout($ip);
        if ($lockout) {
            return new WP_Error('locked_out', 'Too many attempts. Please try again later.');
        }

        return true;
    }

    /**
     * Before magic link sent
     *
     * @param string $email Email address
     */
    public function before_magic_link($email) {
        $ip = $this->ip_manager->get_client_ip();

        // Rate limit magic link requests
        $cache_key = 'artistpro_security_magic_' . md5($ip . $email);
        $attempts = get_transient($cache_key) ?: 0;

        if ($attempts >= 3) {
            $this->logger->log([
                'ip_address' => $ip,
                'username' => $email,
                'login_type' => 'portal_email',
                'result' => 'blocked',
                'failure_reason' => 'Rate limited - too many magic link requests'
            ]);

            wp_die('Too many magic link requests. Please wait a few minutes and try again.');
        }

        set_transient($cache_key, $attempts + 1, 5 * MINUTE_IN_SECONDS);
    }

    /**
     * Before SMS code sent
     *
     * @param string $phone Phone number
     */
    public function before_sms($phone) {
        $ip = $this->ip_manager->get_client_ip();

        // Rate limit SMS requests (more strict due to cost)
        $cache_key = 'artistpro_security_sms_' . md5($ip);
        $attempts = get_transient($cache_key) ?: 0;

        if ($attempts >= 5) {
            $this->logger->log([
                'ip_address' => $ip,
                'username' => $phone,
                'login_type' => 'portal_sms',
                'result' => 'blocked',
                'failure_reason' => 'Rate limited - too many SMS requests'
            ]);

            wp_die('Too many SMS code requests. Please wait and try again later.');
        }

        set_transient($cache_key, $attempts + 1, 15 * MINUTE_IN_SECONDS);
    }

    /**
     * Handle portal login result
     *
     * @param int $company_id Company ID (0 if failed)
     * @param string $method Auth method
     * @param bool $success Whether login succeeded
     */
    public function on_portal_login($company_id, $method, $success) {
        $ip = $this->ip_manager->get_client_ip();

        if ($success) {
            // Log successful login
            $this->logger->log([
                'ip_address' => $ip,
                'username' => "Company #{$company_id}",
                'login_type' => 'portal_' . $method,
                'result' => 'success',
                'failure_reason' => null
            ]);

            // Clear any failed attempts
            $this->security_core->clear_failed_attempts($ip);
        } else {
            // Log failed attempt
            $this->logger->log([
                'ip_address' => $ip,
                'username' => 'Unknown',
                'login_type' => 'portal_' . $method,
                'result' => 'failed',
                'failure_reason' => 'Invalid credentials'
            ]);

            // Record failed attempt
            $this->security_core->record_failed_attempt($ip, 'Unknown');
        }
    }

    /**
     * Add security info to portal login page
     *
     * @return string HTML to display
     */
    public function get_login_security_info() {
        $ip = $this->ip_manager->get_client_ip();
        $lockout = $this->security_core->get_active_lockout($ip);

        if (!$lockout) {
            return '';
        }

        $remaining = strtotime($lockout->locked_until) - time();
        $minutes = ceil($remaining / 60);

        return sprintf(
            '<div class="security-notice" style="background:#fff3cd;border:1px solid #ffc107;padding:15px;border-radius:8px;margin-bottom:20px;">
                <strong>⚠️ Account Temporarily Locked</strong><br>
                Too many failed login attempts. Please try again in %d minute%s.
            </div>',
            $minutes,
            $minutes === 1 ? '' : 's'
        );
    }

    /**
     * Check if portal should show captcha
     *
     * @return bool
     */
    public function should_show_captcha() {
        $ip = $this->ip_manager->get_client_ip();

        // Get failed attempts in last hour
        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_logs';

        $failures = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table}
             WHERE ip_address = %s
               AND result = 'failed'
               AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)",
            $ip
        ));

        return $failures >= 2;
    }
}
