<?php
/**
 * Login Logger - Tracks all login attempts
 *
 * @package ArtistPro_Security
 * @since 1.0.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_Login_Logger {

    /**
     * Initialize the class
     */
    public static function init() {
        // No hooks needed - utility class called by Security_Core
    }

    /**
     * Log a login attempt
     *
     * @param array $data Login attempt data
     * @return int|false Insert ID or false
     */
    public static function log_attempt($data) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOGS;

        $defaults = array(
            'ip_address' => ArtistPro_IP_Manager::get_client_ip(),
            'username' => null,
            'email' => null,
            'login_type' => 'wp_login',
            'result' => 'failed',
            'failure_reason' => null,
            'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? substr($_SERVER['HTTP_USER_AGENT'], 0, 500) : null,
            'country_code' => null,
            'city' => null,
        );

        $data = wp_parse_args($data, $defaults);

        // Try to get country code from GeoIP if not provided
        if (empty($data['country_code']) && class_exists('ArtistPro_Geo_Blocker')) {
            try {
                $geo_blocker = new ArtistPro_Geo_Blocker();
                $country = $geo_blocker->get_country($data['ip_address']);
                if ($country) {
                    $data['country_code'] = $country;
                }
            } catch (Exception $e) {
                // GeoIP lookup failed, continue without it
            }
        }

        $result = $wpdb->insert(
            $table,
            array(
                'ip_address' => $data['ip_address'],
                'username' => $data['username'],
                'email' => $data['email'],
                'login_type' => $data['login_type'],
                'result' => $data['result'],
                'failure_reason' => $data['failure_reason'],
                'user_agent' => $data['user_agent'],
                'country_code' => $data['country_code'],
                'city' => $data['city'],
            ),
            array('%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s')
        );

        if ($result) {
            // Update stats
            self::update_stats($data['result']);
            return $wpdb->insert_id;
        }

        return false;
    }

    /**
     * Update hourly statistics
     *
     * @param string $result Login result
     */
    private static function update_stats($result) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_STATS;

        $date = current_time('Y-m-d');
        $hour = (int) current_time('G');

        // Try to update existing record
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table WHERE stat_date = %s AND stat_hour = %d",
            $date,
            $hour
        ));

        $field_map = array(
            'success' => 'successful_logins',
            'failed' => 'failed_attempts',
            'blocked' => 'blocked_attempts',
            'lockout' => 'lockouts_triggered',
        );

        $field = isset($field_map[$result]) ? $field_map[$result] : 'total_attempts';

        if ($existing) {
            $wpdb->query($wpdb->prepare(
                "UPDATE $table SET total_attempts = total_attempts + 1, $field = $field + 1 WHERE id = %d",
                $existing
            ));
        } else {
            $wpdb->insert(
                $table,
                array(
                    'stat_date' => $date,
                    'stat_hour' => $hour,
                    'total_attempts' => 1,
                    'failed_attempts' => $result === 'failed' ? 1 : 0,
                    'successful_logins' => $result === 'success' ? 1 : 0,
                    'blocked_attempts' => $result === 'blocked' ? 1 : 0,
                    'lockouts_triggered' => $result === 'lockout' ? 1 : 0,
                    'unique_ips' => 1,
                ),
                array('%s', '%d', '%d', '%d', '%d', '%d', '%d', '%d')
            );
        }
    }

    /**
     * Get recent login attempts
     *
     * @param int $limit Number of records
     * @param int $offset Offset
     * @param array $filters Filter criteria
     * @return array
     */
    public static function get_recent_attempts($limit = 100, $offset = 0, $filters = array()) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOGS;

        $where = array('1=1');
        $params = array();

        if (!empty($filters['ip_address'])) {
            $where[] = 'ip_address = %s';
            $params[] = $filters['ip_address'];
        }

        if (!empty($filters['username'])) {
            $where[] = 'username LIKE %s';
            $params[] = '%' . $wpdb->esc_like($filters['username']) . '%';
        }

        if (!empty($filters['result'])) {
            $where[] = 'result = %s';
            $params[] = $filters['result'];
        }

        if (!empty($filters['login_type'])) {
            $where[] = 'login_type = %s';
            $params[] = $filters['login_type'];
        }

        if (!empty($filters['country_code'])) {
            $where[] = 'country_code = %s';
            $params[] = $filters['country_code'];
        }

        if (!empty($filters['date_from'])) {
            $where[] = 'created_at >= %s';
            $params[] = $filters['date_from'] . ' 00:00:00';
        }

        if (!empty($filters['date_to'])) {
            $where[] = 'created_at <= %s';
            $params[] = $filters['date_to'] . ' 23:59:59';
        }

        $where_clause = implode(' AND ', $where);
        $params[] = $limit;
        $params[] = $offset;

        $sql = "SELECT * FROM $table WHERE $where_clause ORDER BY created_at DESC LIMIT %d OFFSET %d";

        if (!empty($params)) {
            return $wpdb->get_results($wpdb->prepare($sql, $params));
        }

        return $wpdb->get_results($sql);
    }

    /**
     * Get total count of login attempts with filters
     *
     * @param array $filters Filter criteria
     * @return int
     */
    public static function get_attempts_count($filters = array()) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOGS;

        $where = array('1=1');
        $params = array();

        if (!empty($filters['ip_address'])) {
            $where[] = 'ip_address = %s';
            $params[] = $filters['ip_address'];
        }

        if (!empty($filters['username'])) {
            $where[] = 'username LIKE %s';
            $params[] = '%' . $wpdb->esc_like($filters['username']) . '%';
        }

        if (!empty($filters['result'])) {
            $where[] = 'result = %s';
            $params[] = $filters['result'];
        }

        if (!empty($filters['login_type'])) {
            $where[] = 'login_type = %s';
            $params[] = $filters['login_type'];
        }

        if (!empty($filters['date_from'])) {
            $where[] = 'created_at >= %s';
            $params[] = $filters['date_from'] . ' 00:00:00';
        }

        if (!empty($filters['date_to'])) {
            $where[] = 'created_at <= %s';
            $params[] = $filters['date_to'] . ' 23:59:59';
        }

        $where_clause = implode(' AND ', $where);

        if (!empty($params)) {
            return (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table WHERE $where_clause",
                $params
            ));
        }

        return (int) $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE $where_clause");
    }

    /**
     * Get attempts by IP in a time window
     *
     * @param string $ip IP address
     * @param int $seconds Seconds to look back
     * @return int
     */
    public static function get_attempts_by_ip($ip, $seconds = 3600) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOGS;

        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table
             WHERE ip_address = %s
             AND result = 'failed'
             AND created_at >= DATE_SUB(NOW(), INTERVAL %d SECOND)",
            $ip,
            $seconds
        ));
    }

    /**
     * Get attempts by username in a time window
     *
     * @param string $username Username
     * @param int $seconds Seconds to look back
     * @return int
     */
    public static function get_attempts_by_username($username, $seconds = 3600) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOGS;

        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table
             WHERE username = %s
             AND result = 'failed'
             AND created_at >= DATE_SUB(NOW(), INTERVAL %d SECOND)",
            $username,
            $seconds
        ));
    }

    /**
     * Get statistics for a date range
     *
     * @param string $from Start date (Y-m-d)
     * @param string $to End date (Y-m-d)
     * @return array
     */
    public static function get_stats($from = null, $to = null) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_STATS;

        if (!$from) {
            $from = date('Y-m-d', strtotime('-7 days'));
        }
        if (!$to) {
            $to = date('Y-m-d');
        }

        return $wpdb->get_results($wpdb->prepare(
            "SELECT
                stat_date,
                SUM(total_attempts) as total_attempts,
                SUM(failed_attempts) as failed_attempts,
                SUM(successful_logins) as successful_logins,
                SUM(blocked_attempts) as blocked_attempts,
                SUM(lockouts_triggered) as lockouts_triggered
             FROM $table
             WHERE stat_date BETWEEN %s AND %s
             GROUP BY stat_date
             ORDER BY stat_date ASC",
            $from,
            $to
        ));
    }

    /**
     * Get summary statistics
     *
     * @param int $hours Hours to look back (default 24)
     * @return array
     */
    public static function get_summary($hours = 24) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOGS;

        $since = date('Y-m-d H:i:s', strtotime("-{$hours} hours"));

        return $wpdb->get_row($wpdb->prepare(
            "SELECT
                COUNT(*) as total_attempts,
                SUM(result = 'failed') as failed_attempts,
                SUM(result = 'success') as successful_logins,
                SUM(result = 'blocked') as blocked_attempts,
                SUM(result = 'lockout') as lockouts,
                COUNT(DISTINCT ip_address) as unique_ips
             FROM $table
             WHERE created_at >= %s",
            $since
        ), ARRAY_A);
    }

    /**
     * Get top offending IPs
     *
     * @param int $limit Number of IPs
     * @param int $hours Hours to look back
     * @return array
     */
    public static function get_top_ips($limit = 10, $hours = 24) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOGS;

        $since = date('Y-m-d H:i:s', strtotime("-{$hours} hours"));

        return $wpdb->get_results($wpdb->prepare(
            "SELECT
                ip_address,
                country_code,
                COUNT(*) as attempt_count,
                SUM(result = 'failed') as failed_count,
                SUM(result = 'blocked') as blocked_count,
                MAX(created_at) as last_attempt
             FROM $table
             WHERE created_at >= %s
             AND result IN ('failed', 'blocked', 'lockout')
             GROUP BY ip_address, country_code
             ORDER BY attempt_count DESC
             LIMIT %d",
            $since,
            $limit
        ));
    }

    /**
     * Clean up old log entries
     *
     * @param int $days Days to keep logs
     * @return int Number of deleted rows
     */
    public static function cleanup_old_logs($days = null) {
        global $wpdb;

        if (!$days) {
            $days = get_option('artistpro_security_log_retention_days', 30);
        }

        $table_logs = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOGS;
        $table_stats = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_STATS;

        $deleted_logs = $wpdb->query($wpdb->prepare(
            "DELETE FROM $table_logs WHERE created_at < DATE_SUB(NOW(), INTERVAL %d DAY)",
            $days
        ));

        // Keep stats for longer (90 days)
        $wpdb->query($wpdb->prepare(
            "DELETE FROM $table_stats WHERE stat_date < DATE_SUB(NOW(), INTERVAL %d DAY)",
            $days * 3
        ));

        return $deleted_logs;
    }

    /**
     * Export logs to CSV format
     *
     * @param array $filters Filter criteria
     * @return string CSV content
     */
    public static function export_csv($filters = array()) {
        $logs = self::get_recent_attempts(10000, 0, $filters);

        $csv = "ID,IP Address,Username,Email,Login Type,Result,Failure Reason,User Agent,Country,City,Date\n";

        foreach ($logs as $log) {
            $csv .= sprintf(
                "%d,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s\n",
                $log->id,
                $log->ip_address,
                $log->username ?? '',
                $log->email ?? '',
                $log->login_type,
                $log->result,
                str_replace(',', ';', $log->failure_reason ?? ''),
                str_replace(',', ';', substr($log->user_agent ?? '', 0, 100)),
                $log->country_code ?? '',
                $log->city ?? '',
                $log->created_at
            );
        }

        return $csv;
    }
}
