<?php
/**
 * License & Update Checker for artistproSecurity
 * Connects to artistprowp.com API
 *
 * @package ArtistPro_Security
 * @since 1.1.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_Security_Updater {

    private $api_url = 'https://artistprowp.com/wp-json/artistpro-license/v1/';
    private $product_id = 'artistprosecurity';
    private $version;
    private $plugin_file;

    public function __construct() {
        $this->version = ARTISTPRO_SECURITY_VERSION;
        $this->plugin_file = ARTISTPRO_SECURITY_BASENAME;

        add_filter('pre_set_site_transient_update_plugins', array($this, 'check_update'));
        add_filter('plugins_api', array($this, 'plugin_info'), 10, 3);
        
        // Admin notices
        add_action('admin_notices', array($this, 'license_notice'));
    }

    /**
     * Check for plugin updates
     */
    public function check_update($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }

        $license_key = get_option('artistpro_security_license_key');

        if (empty($license_key)) {
            return $transient; // No license, no updates
        }

        // Call API
        $response = wp_remote_post($this->api_url . 'update-check', array(
            'body' => array(
                'license_key' => $license_key,
                'product_id' => $this->product_id,
                'version' => $this->version,
                'site_url' => home_url(),
            ),
            'timeout' => 15,
        ));

        if (is_wp_error($response)) {
            return $transient;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($data['update_available'])) {
            $plugin_data = array(
                'slug' => 'artistpro-security',
                'new_version' => $data['new_version'],
                'package' => $data['package'],
                'url' => 'https://artistprowp.com',
            );

            $transient->response[$this->plugin_file] = (object) $plugin_data;
        }

        return $transient;
    }

    /**
     * Plugin info for update screen
     */
    public function plugin_info($false, $action, $args) {
        if ($action !== 'plugin_information' || $args->slug !== 'artistpro-security') {
            return $false;
        }

        return (object) array(
            'name' => 'artistpro Security',
            'slug' => 'artistpro-security',
            'version' => $this->version,
            'author' => 'artistpro Media',
            'homepage' => 'https://artistprowp.com',
            'sections' => array(
                'description' => 'Premium WordPress security with Cloudflare edge blocking, 2FA, and reCAPTCHA.',
            ),
        );
    }

    /**
     * Show license activation notice
     */
    public function license_notice() {
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'artistpro-security') === false) {
            return;
        }

        $license_key = get_option('artistpro_security_license_key');
        $license_status = get_option('artistpro_security_license_status');

        if (empty($license_key)) {
            ?>
            <div class="notice notice-warning">
                <p><strong>artistpro Security:</strong> Please enter your license key to receive updates. 
                <a href="?page=artistpro-security&tab=settings#license">Enter License Key</a></p>
            </div>
            <?php
        } elseif ($license_status !== 'active') {
            ?>
            <div class="notice notice-error">
                <p><strong>artistpro Security:</strong> Your license is <?php echo esc_html($license_status ?: 'inactive'); ?>. 
                <a href="https://artistprowp.com/my-account/">Manage License</a></p>
            </div>
            <?php
        }
    }

    /**
     * Activate license
     *
     * @param string $license_key License key
     * @return array Result
     */
    public static function activate_license($license_key) {
        $response = wp_remote_post('https://artistprowp.com/wp-json/artistpro-license/v1/activate', array(
            'body' => array(
                'license_key' => $license_key,
                'product_id' => 'artistprosecurity',
                'site_url' => home_url(),
            ),
            'timeout' => 15,
        ));

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => 'Connection error: ' . $response->get_error_message(),
            );
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($data['success'])) {
            update_option('artistpro_security_license_key', $license_key);
            update_option('artistpro_security_license_status', 'active');
            update_option('artistpro_security_license_data', $data['license']);
        }

        return $data;
    }

    /**
     * Deactivate license
     *
     * @param string $license_key License key
     * @return array Result
     */
    public static function deactivate_license($license_key) {
        $response = wp_remote_post('https://artistprowp.com/wp-json/artistpro-license/v1/deactivate', array(
            'body' => array(
                'license_key' => $license_key,
                'site_url' => home_url(),
            ),
            'timeout' => 15,
        ));

        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => 'Connection error: ' . $response->get_error_message(),
            );
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($data['success'])) {
            delete_option('artistpro_security_license_key');
            delete_option('artistpro_security_license_status');
            delete_option('artistpro_security_license_data');
        }

        return $data;
    }
}

// Initialize
if (class_exists('ArtistPro_Security_Updater')) {
    new ArtistPro_Security_Updater();
}
