<?php
/**
 * IP Manager - Detection, whitelist/blacklist management
 *
 * @package ArtistPro_Security
 * @since 1.0.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_IP_Manager {

    /**
     * Initialize the class
     */
    public static function init() {
        // No hooks needed - utility class
    }

    /**
     * Get the client's IP address with proxy support
     *
     * @return string IP address
     */
    public static function get_client_ip() {
        $ip = '';

        // Check trusted proxies setting
        $trusted_proxies = get_option('artistpro_security_trusted_proxies', '');
        $proxy_header = get_option('artistpro_security_proxy_header', 'HTTP_X_FORWARDED_FOR');

        // Priority order for IP detection
        $headers = array(
            $proxy_header,
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'HTTP_CF_CONNECTING_IP', // Cloudflare
            'HTTP_CLIENT_IP',
            'REMOTE_ADDR',
        );

        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip_list = $_SERVER[$header];

                // Handle comma-separated list (X-Forwarded-For can have multiple IPs)
                if (strpos($ip_list, ',') !== false) {
                    $ips = array_map('trim', explode(',', $ip_list));
                    // Get the first non-private IP (client's actual IP)
                    foreach ($ips as $potential_ip) {
                        if (self::is_valid_ip($potential_ip) && !self::is_private_ip($potential_ip)) {
                            $ip = $potential_ip;
                            break;
                        }
                    }
                    // If all are private, use the first one
                    if (empty($ip) && !empty($ips[0]) && self::is_valid_ip($ips[0])) {
                        $ip = $ips[0];
                    }
                } else {
                    if (self::is_valid_ip($ip_list)) {
                        $ip = $ip_list;
                    }
                }

                if (!empty($ip)) {
                    break;
                }
            }
        }

        // Fallback to REMOTE_ADDR
        if (empty($ip) && !empty($_SERVER['REMOTE_ADDR'])) {
            $ip = $_SERVER['REMOTE_ADDR'];
        }

        // Final validation
        if (!self::is_valid_ip($ip)) {
            $ip = '0.0.0.0';
        }

        return $ip;
    }

    /**
     * Validate an IP address
     *
     * @param string $ip IP address
     * @return bool
     */
    public static function is_valid_ip($ip) {
        return filter_var($ip, FILTER_VALIDATE_IP) !== false;
    }

    /**
     * Check if IP is in a private range
     *
     * @param string $ip IP address
     * @return bool
     */
    public static function is_private_ip($ip) {
        return filter_var(
            $ip,
            FILTER_VALIDATE_IP,
            FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE
        ) === false;
    }

    /**
     * Check if IP is whitelisted
     *
     * @param string $ip IP address
     * @return bool
     */
    public static function is_whitelisted($ip) {
        return self::check_ip_in_list($ip, 'whitelist');
    }

    /**
     * Check if IP is blacklisted
     *
     * @param string $ip IP address
     * @return bool
     */
    public static function is_blacklisted($ip) {
        return self::check_ip_in_list($ip, 'blacklist');
    }

    /**
     * Check if IP is in a list (whitelist or blacklist)
     *
     * @param string $ip IP address
     * @param string $list_type 'whitelist' or 'blacklist'
     * @return bool
     */
    private static function check_ip_in_list($ip, $list_type) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_IP_LISTS;

        // First check exact IP match
        $exact_match = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table
             WHERE ip_address = %s
             AND list_type = %s
             AND (expires_at IS NULL OR expires_at > NOW())",
            $ip,
            $list_type
        ));

        if ($exact_match > 0) {
            return true;
        }

        // Check CIDR ranges
        $cidr_entries = $wpdb->get_results($wpdb->prepare(
            "SELECT cidr_notation FROM $table
             WHERE cidr_notation IS NOT NULL
             AND list_type = %s
             AND (expires_at IS NULL OR expires_at > NOW())",
            $list_type
        ));

        foreach ($cidr_entries as $entry) {
            if (self::ip_in_cidr($ip, $entry->cidr_notation)) {
                return true;
            }
        }

        // Check IP ranges
        $range_entries = $wpdb->get_results($wpdb->prepare(
            "SELECT ip_range_start, ip_range_end FROM $table
             WHERE ip_range_start IS NOT NULL
             AND ip_range_end IS NOT NULL
             AND list_type = %s
             AND (expires_at IS NULL OR expires_at > NOW())",
            $list_type
        ));

        foreach ($range_entries as $entry) {
            if (self::ip_in_range($ip, $entry->ip_range_start, $entry->ip_range_end)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if IP is in CIDR range
     *
     * @param string $ip IP address
     * @param string $cidr CIDR notation (e.g., 192.168.1.0/24)
     * @return bool
     */
    public static function ip_in_cidr($ip, $cidr) {
        if (strpos($cidr, '/') === false) {
            return $ip === $cidr;
        }

        list($subnet, $bits) = explode('/', $cidr);

        // Handle IPv6
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            return self::ipv6_in_cidr($ip, $subnet, (int)$bits);
        }

        // IPv4
        $ip_long = ip2long($ip);
        $subnet_long = ip2long($subnet);

        if ($ip_long === false || $subnet_long === false) {
            return false;
        }

        $mask = -1 << (32 - (int)$bits);
        $subnet_long &= $mask;

        return ($ip_long & $mask) === $subnet_long;
    }

    /**
     * Check if IPv6 is in CIDR range
     *
     * @param string $ip IPv6 address
     * @param string $subnet Subnet
     * @param int $bits Prefix length
     * @return bool
     */
    private static function ipv6_in_cidr($ip, $subnet, $bits) {
        $ip_bin = inet_pton($ip);
        $subnet_bin = inet_pton($subnet);

        if ($ip_bin === false || $subnet_bin === false) {
            return false;
        }

        $ip_hex = bin2hex($ip_bin);
        $subnet_hex = bin2hex($subnet_bin);

        $ip_bits = '';
        $subnet_bits = '';

        for ($i = 0; $i < strlen($ip_hex); $i++) {
            $ip_bits .= str_pad(base_convert($ip_hex[$i], 16, 2), 4, '0', STR_PAD_LEFT);
            $subnet_bits .= str_pad(base_convert($subnet_hex[$i], 16, 2), 4, '0', STR_PAD_LEFT);
        }

        return substr($ip_bits, 0, $bits) === substr($subnet_bits, 0, $bits);
    }

    /**
     * Check if IP is in a numeric range
     *
     * @param string $ip IP address
     * @param string $start Range start
     * @param string $end Range end
     * @return bool
     */
    public static function ip_in_range($ip, $start, $end) {
        $ip_long = ip2long($ip);
        $start_long = ip2long($start);
        $end_long = ip2long($end);

        if ($ip_long === false || $start_long === false || $end_long === false) {
            return false;
        }

        return $ip_long >= $start_long && $ip_long <= $end_long;
    }

    /**
     * Add IP to whitelist
     *
     * @param string $ip IP address or CIDR
     * @param string $reason Reason for whitelisting
     * @return int|false Insert ID or false
     */
    public static function add_to_whitelist($ip, $reason = '') {
        return self::add_to_list($ip, 'whitelist', 'manual', $reason);
    }

    /**
     * Add IP to blacklist
     *
     * @param string $ip IP address or CIDR
     * @param string $reason Reason for blacklisting
     * @param string $source Source (manual, firehol, spamhaus, etc.)
     * @param string|null $expires Expiration datetime or null for permanent
     * @return int|false Insert ID or false
     */
    public static function add_to_blacklist($ip, $reason = '', $source = 'manual', $expires = null) {
        return self::add_to_list($ip, 'blacklist', $source, $reason, $expires);
    }

    /**
     * Add IP to a list
     *
     * @param string $ip IP address or CIDR
     * @param string $list_type 'whitelist' or 'blacklist'
     * @param string $source Source of entry
     * @param string $reason Reason
     * @param string|null $expires Expiration datetime
     * @return int|false Insert ID or false
     */
    private static function add_to_list($ip, $list_type, $source, $reason, $expires = null) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_IP_LISTS;

        $data = array(
            'list_type' => $list_type,
            'source' => $source,
            'reason' => $reason,
        );

        $format = array('%s', '%s', '%s');

        // Determine if it's a CIDR, range, or single IP
        if (strpos($ip, '/') !== false) {
            // CIDR notation
            $data['ip_address'] = explode('/', $ip)[0];
            $data['cidr_notation'] = $ip;
            $format[] = '%s';
            $format[] = '%s';
        } elseif (strpos($ip, '-') !== false) {
            // IP range
            list($start, $end) = array_map('trim', explode('-', $ip));
            $data['ip_address'] = $start;
            $data['ip_range_start'] = $start;
            $data['ip_range_end'] = $end;
            $format[] = '%s';
            $format[] = '%s';
            $format[] = '%s';
        } else {
            // Single IP
            $data['ip_address'] = $ip;
            $format[] = '%s';
        }

        if ($expires) {
            $data['expires_at'] = $expires;
            $format[] = '%s';
        }

        // Check if already exists
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table WHERE ip_address = %s AND list_type = %s",
            $data['ip_address'],
            $list_type
        ));

        if ($existing) {
            // Update existing
            $wpdb->update(
                $table,
                $data,
                array('id' => $existing),
                $format,
                array('%d')
            );
            return $existing;
        }

        $result = $wpdb->insert($table, $data, $format);

        return $result ? $wpdb->insert_id : false;
    }

    /**
     * Remove IP from a list
     *
     * @param string $ip IP address
     * @param string $list_type 'whitelist' or 'blacklist'
     * @return bool
     */
    public static function remove_from_list($ip, $list_type) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_IP_LISTS;

        return $wpdb->delete(
            $table,
            array(
                'ip_address' => $ip,
                'list_type' => $list_type,
            ),
            array('%s', '%s')
        ) !== false;
    }

    /**
     * Get all entries from a list
     *
     * @param string $list_type 'whitelist' or 'blacklist'
     * @param string|null $source Filter by source
     * @param int $limit Limit results
     * @param int $offset Offset
     * @return array
     */
    public static function get_list($list_type, $source = null, $limit = 100, $offset = 0) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_IP_LISTS;

        $where = array("list_type = %s");
        $params = array($list_type);

        if ($source) {
            $where[] = "source = %s";
            $params[] = $source;
        }

        $where_clause = implode(' AND ', $where);
        $params[] = $limit;
        $params[] = $offset;

        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table WHERE $where_clause ORDER BY created_at DESC LIMIT %d OFFSET %d",
            $params
        ));
    }

    /**
     * Get count of entries in a list
     *
     * @param string $list_type 'whitelist' or 'blacklist'
     * @param string|null $source Filter by source
     * @return int
     */
    public static function get_list_count($list_type, $source = null) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_IP_LISTS;

        if ($source) {
            return (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table WHERE list_type = %s AND source = %s",
                $list_type,
                $source
            ));
        }

        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table WHERE list_type = %s",
            $list_type
        ));
    }

    /**
     * Clear all entries from a source
     *
     * @param string $source Source name
     * @return int Number of deleted rows
     */
    public static function clear_source($source) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_IP_LISTS;

        return $wpdb->delete(
            $table,
            array('source' => $source),
            array('%s')
        );
    }

    /**
     * Clean up expired entries
     *
     * @return int Number of deleted rows
     */
    public static function cleanup_expired() {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_IP_LISTS;

        return $wpdb->query(
            "DELETE FROM $table WHERE expires_at IS NOT NULL AND expires_at < NOW()"
        );
    }
}
