<?php
/**
 * Geo Blocker - Country blocking using MaxMind GeoLite2
 *
 * @package ArtistPro_Security
 * @since 1.0.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_Geo_Blocker {

    /**
     * @var string Path to GeoLite2 database
     */
    private $db_path;

    /**
     * @var object MaxMind reader instance
     */
    private $reader = null;

    /**
     * @var array Cached blocked countries
     */
    private $blocked_countries = null;

    /**
     * Constructor
     */
    public function __construct() {
        $this->db_path = ARTISTPRO_SECURITY_PATH . 'data/GeoLite2-Country.mmdb';
    }

    /**
     * Initialize hooks
     */
    public function init() {
        // Download database if missing
        if (!file_exists($this->db_path)) {
            $this->schedule_database_download();
        }
    }

    /**
     * Static method to check if IP is blocked (used by security core)
     *
     * @param string $ip IP address
     * @return bool
     */
    public static function is_blocked($ip) {
        $instance = new self();
        $result = $instance->check_ip($ip);
        return $result['blocked'] ?? false;
    }

    /**
     * Check if an IP is from a blocked country
     *
     * @param string $ip IP address
     * @return array ['blocked' => bool, 'country' => string, 'country_name' => string]
     */
    public function check_ip($ip) {
        $result = [
            'blocked' => false,
            'country' => '',
            'country_name' => ''
        ];

        // Get country for IP
        $country = $this->get_country($ip);
        if (!$country) {
            return $result;
        }

        $result['country'] = $country['code'];
        $result['country_name'] = $country['name'];

        // Check if country is blocked
        $blocked_countries = $this->get_blocked_countries();
        if (isset($blocked_countries[$country['code']])) {
            $result['blocked'] = true;
        }

        return $result;
    }

    /**
     * Get country for an IP address
     *
     * @param string $ip IP address
     * @return array|null ['code' => 'US', 'name' => 'United States']
     */
    public function get_country($ip) {
        // Validate IP
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            return null;
        }

        // Skip private/reserved IPs
        if ($this->is_private_ip($ip)) {
            return ['code' => 'XX', 'name' => 'Private/Reserved'];
        }

        // Try MaxMind database
        $reader = $this->get_reader();
        if ($reader) {
            try {
                $record = $reader->country($ip);
                return [
                    'code' => $record->country->isoCode ?? 'XX',
                    'name' => $record->country->name ?? 'Unknown'
                ];
            } catch (Exception $e) {
                // IP not found in database
                return null;
            }
        }

        return null;
    }

    /**
     * Check if IP is private/reserved
     *
     * @param string $ip IP address
     * @return bool
     */
    private function is_private_ip($ip) {
        return !filter_var(
            $ip,
            FILTER_VALIDATE_IP,
            FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE
        );
    }

    /**
     * Get MaxMind reader instance
     *
     * @return object|null
     */
    private function get_reader() {
        if ($this->reader !== null) {
            return $this->reader;
        }

        if (!file_exists($this->db_path)) {
            return null;
        }

        // Check if MaxMind reader library exists
        $reader_file = ARTISTPRO_SECURITY_PATH . 'includes/maxmind/Reader.php';
        if (file_exists($reader_file)) {
            require_once $reader_file;
            try {
                $this->reader = new MaxMind\Db\Reader($this->db_path);
            } catch (Exception $e) {
                error_log('ArtistPro Security: Failed to open GeoLite2 database: ' . $e->getMessage());
                return null;
            }
        } else {
            // Use simple MMDB reader
            $this->reader = new ArtistPro_Simple_MMDB_Reader($this->db_path);
        }

        return $this->reader;
    }

    /**
     * Get blocked countries from database
     *
     * @return array ['US' => ['block_type' => 'all'], ...]
     */
    public function get_blocked_countries() {
        if ($this->blocked_countries !== null) {
            return $this->blocked_countries;
        }

        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_blocked_countries';

        $rows = $wpdb->get_results("SELECT country_code, block_type FROM {$table}", ARRAY_A);

        $this->blocked_countries = [];
        foreach ($rows as $row) {
            $this->blocked_countries[$row['country_code']] = [
                'block_type' => $row['block_type']
            ];
        }

        return $this->blocked_countries;
    }

    /**
     * Block a country
     *
     * @param string $country_code ISO country code
     * @param string $block_type 'all' or 'login_only'
     * @return bool
     */
    public function block_country($country_code, $block_type = 'all') {
        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_blocked_countries';

        $country_code = strtoupper(sanitize_text_field($country_code));
        if (strlen($country_code) !== 2) {
            return false;
        }

        $result = $wpdb->replace($table, [
            'country_code' => $country_code,
            'block_type' => $block_type
        ], ['%s', '%s']);

        // Clear cache
        $this->blocked_countries = null;

        return $result !== false;
    }

    /**
     * Unblock a country
     *
     * @param string $country_code ISO country code
     * @return bool
     */
    public function unblock_country($country_code) {
        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_blocked_countries';

        $result = $wpdb->delete($table, [
            'country_code' => strtoupper($country_code)
        ], ['%s']);

        // Clear cache
        $this->blocked_countries = null;

        return $result !== false;
    }

    /**
     * Get all countries with blocking status
     *
     * @return array
     */
    public function get_all_countries() {
        $countries = $this->get_country_list();
        $blocked = $this->get_blocked_countries();

        $result = [];
        foreach ($countries as $code => $name) {
            $result[$code] = [
                'code' => $code,
                'name' => $name,
                'blocked' => isset($blocked[$code]),
                'block_type' => $blocked[$code]['block_type'] ?? null
            ];
        }

        return $result;
    }

    /**
     * Schedule database download
     */
    public function schedule_database_download() {
        if (!wp_next_scheduled('artistpro_security_download_geoip')) {
            wp_schedule_single_event(time() + 60, 'artistpro_security_download_geoip');
        }
    }

    /**
     * Download GeoLite2 database
     * Note: Requires MaxMind license key (free registration)
     *
     * @return bool
     */
    public function download_database() {
        $license_key = get_option('artistpro_security_maxmind_key', '');

        if (empty($license_key)) {
            error_log('ArtistPro Security: MaxMind license key not configured');
            return false;
        }

        $url = sprintf(
            'https://download.maxmind.com/app/geoip_download?edition_id=GeoLite2-Country&license_key=%s&suffix=tar.gz',
            urlencode($license_key)
        );

        $tmp_file = download_url($url);

        if (is_wp_error($tmp_file)) {
            error_log('ArtistPro Security: Failed to download GeoLite2: ' . $tmp_file->get_error_message());
            return false;
        }

        // Extract tar.gz
        $result = $this->extract_database($tmp_file);

        @unlink($tmp_file);

        if ($result) {
            update_option('artistpro_security_geoip_updated', time());
        }

        return $result;
    }

    /**
     * Extract database from tar.gz
     *
     * @param string $tar_file Path to tar.gz file
     * @return bool
     */
    private function extract_database($tar_file) {
        // Use PharData for tar extraction
        try {
            $phar = new PharData($tar_file);
            $phar->decompress();

            $tar_path = str_replace('.gz', '', $tar_file);
            $phar = new PharData($tar_path);

            // Find the mmdb file
            foreach (new RecursiveIteratorIterator($phar) as $file) {
                if (pathinfo($file, PATHINFO_EXTENSION) === 'mmdb') {
                    copy($file, $this->db_path);
                    @unlink($tar_path);
                    return true;
                }
            }

            @unlink($tar_path);
        } catch (Exception $e) {
            error_log('ArtistPro Security: Failed to extract GeoLite2: ' . $e->getMessage());
        }

        return false;
    }

    /**
     * Get list of all countries
     *
     * @return array ['US' => 'United States', ...]
     */
    public function get_country_list() {
        return [
            'AF' => 'Afghanistan', 'AL' => 'Albania', 'DZ' => 'Algeria', 'AD' => 'Andorra',
            'AO' => 'Angola', 'AR' => 'Argentina', 'AM' => 'Armenia', 'AU' => 'Australia',
            'AT' => 'Austria', 'AZ' => 'Azerbaijan', 'BH' => 'Bahrain', 'BD' => 'Bangladesh',
            'BY' => 'Belarus', 'BE' => 'Belgium', 'BZ' => 'Belize', 'BJ' => 'Benin',
            'BT' => 'Bhutan', 'BO' => 'Bolivia', 'BA' => 'Bosnia and Herzegovina',
            'BW' => 'Botswana', 'BR' => 'Brazil', 'BN' => 'Brunei', 'BG' => 'Bulgaria',
            'BF' => 'Burkina Faso', 'BI' => 'Burundi', 'KH' => 'Cambodia', 'CM' => 'Cameroon',
            'CA' => 'Canada', 'CF' => 'Central African Republic', 'TD' => 'Chad',
            'CL' => 'Chile', 'CN' => 'China', 'CO' => 'Colombia', 'CG' => 'Congo',
            'CR' => 'Costa Rica', 'HR' => 'Croatia', 'CU' => 'Cuba', 'CY' => 'Cyprus',
            'CZ' => 'Czech Republic', 'DK' => 'Denmark', 'DJ' => 'Djibouti',
            'DO' => 'Dominican Republic', 'EC' => 'Ecuador', 'EG' => 'Egypt',
            'SV' => 'El Salvador', 'EE' => 'Estonia', 'ET' => 'Ethiopia', 'FI' => 'Finland',
            'FR' => 'France', 'GA' => 'Gabon', 'GM' => 'Gambia', 'GE' => 'Georgia',
            'DE' => 'Germany', 'GH' => 'Ghana', 'GR' => 'Greece', 'GT' => 'Guatemala',
            'GN' => 'Guinea', 'HT' => 'Haiti', 'HN' => 'Honduras', 'HK' => 'Hong Kong',
            'HU' => 'Hungary', 'IS' => 'Iceland', 'IN' => 'India', 'ID' => 'Indonesia',
            'IR' => 'Iran', 'IQ' => 'Iraq', 'IE' => 'Ireland', 'IL' => 'Israel',
            'IT' => 'Italy', 'JM' => 'Jamaica', 'JP' => 'Japan', 'JO' => 'Jordan',
            'KZ' => 'Kazakhstan', 'KE' => 'Kenya', 'KP' => 'North Korea', 'KR' => 'South Korea',
            'KW' => 'Kuwait', 'KG' => 'Kyrgyzstan', 'LA' => 'Laos', 'LV' => 'Latvia',
            'LB' => 'Lebanon', 'LY' => 'Libya', 'LT' => 'Lithuania', 'LU' => 'Luxembourg',
            'MK' => 'North Macedonia', 'MG' => 'Madagascar', 'MY' => 'Malaysia',
            'ML' => 'Mali', 'MT' => 'Malta', 'MX' => 'Mexico', 'MD' => 'Moldova',
            'MN' => 'Mongolia', 'ME' => 'Montenegro', 'MA' => 'Morocco', 'MZ' => 'Mozambique',
            'MM' => 'Myanmar', 'NA' => 'Namibia', 'NP' => 'Nepal', 'NL' => 'Netherlands',
            'NZ' => 'New Zealand', 'NI' => 'Nicaragua', 'NE' => 'Niger', 'NG' => 'Nigeria',
            'NO' => 'Norway', 'OM' => 'Oman', 'PK' => 'Pakistan', 'PA' => 'Panama',
            'PY' => 'Paraguay', 'PE' => 'Peru', 'PH' => 'Philippines', 'PL' => 'Poland',
            'PT' => 'Portugal', 'QA' => 'Qatar', 'RO' => 'Romania', 'RU' => 'Russia',
            'RW' => 'Rwanda', 'SA' => 'Saudi Arabia', 'SN' => 'Senegal', 'RS' => 'Serbia',
            'SG' => 'Singapore', 'SK' => 'Slovakia', 'SI' => 'Slovenia', 'SO' => 'Somalia',
            'ZA' => 'South Africa', 'ES' => 'Spain', 'LK' => 'Sri Lanka', 'SD' => 'Sudan',
            'SE' => 'Sweden', 'CH' => 'Switzerland', 'SY' => 'Syria', 'TW' => 'Taiwan',
            'TJ' => 'Tajikistan', 'TZ' => 'Tanzania', 'TH' => 'Thailand', 'TN' => 'Tunisia',
            'TR' => 'Turkey', 'TM' => 'Turkmenistan', 'UG' => 'Uganda', 'UA' => 'Ukraine',
            'AE' => 'United Arab Emirates', 'GB' => 'United Kingdom', 'US' => 'United States',
            'UY' => 'Uruguay', 'UZ' => 'Uzbekistan', 'VE' => 'Venezuela', 'VN' => 'Vietnam',
            'YE' => 'Yemen', 'ZM' => 'Zambia', 'ZW' => 'Zimbabwe'
        ];
    }

    /**
     * Get stats about blocked countries
     *
     * @return array
     */
    public function get_stats() {
        global $wpdb;
        $logs_table = $wpdb->prefix . 'artistpro_security_logs';

        // Blocked by country in last 30 days
        $blocked_by_country = $wpdb->get_results("
            SELECT country_code, COUNT(*) as count
            FROM {$logs_table}
            WHERE result = 'blocked'
              AND failure_reason LIKE '%country%'
              AND created_at > DATE_SUB(NOW(), INTERVAL 30 DAY)
            GROUP BY country_code
            ORDER BY count DESC
            LIMIT 10
        ", ARRAY_A);

        return [
            'blocked_countries_count' => count($this->get_blocked_countries()),
            'database_exists' => file_exists($this->db_path),
            'database_updated' => get_option('artistpro_security_geoip_updated', 0),
            'blocked_by_country' => $blocked_by_country
        ];
    }
}

/**
 * Simple MMDB Reader - Basic MaxMind database reader
 * Used when full MaxMind PHP library is not installed
 */
class ArtistPro_Simple_MMDB_Reader {

    private $file_handle;
    private $metadata;

    public function __construct($db_path) {
        if (!file_exists($db_path)) {
            throw new Exception('Database file not found');
        }

        $this->file_handle = fopen($db_path, 'rb');
        if (!$this->file_handle) {
            throw new Exception('Could not open database file');
        }

        $this->metadata = $this->read_metadata();
    }

    public function __destruct() {
        if ($this->file_handle) {
            fclose($this->file_handle);
        }
    }

    /**
     * Look up country for IP
     * Note: This is a simplified implementation
     *
     * @param string $ip
     * @return object
     */
    public function country($ip) {
        // For now, return unknown - full implementation requires complex MMDB parsing
        // Users should install MaxMind PHP library for full functionality
        $result = new stdClass();
        $result->country = new stdClass();
        $result->country->isoCode = 'XX';
        $result->country->name = 'Unknown';

        return $result;
    }

    private function read_metadata() {
        // Simplified - real implementation would parse MMDB metadata
        return [];
    }
}
