<?php
/**
 * Emergency Access Recovery System
 * Provides multiple ways to regain access if locked out
 *
 * @package ArtistPro_Security
 * @since 1.1.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_Emergency_Access {

    /**
     * Initialize emergency access system
     */
    public static function init() {
        // Check for emergency access token in URL
        add_action('init', array(__CLASS__, 'check_emergency_access'), 1);
        
        // Add emergency access settings
        add_action('admin_init', array(__CLASS__, 'register_settings'));
        
        // Show emergency access notice to admins
        add_action('admin_notices', array(__CLASS__, 'show_emergency_notice'));
        
        // AJAX handler for generating emergency token
        add_action('wp_ajax_generate_emergency_token', array(__CLASS__, 'ajax_generate_token'));
        
        // Email emergency token
        add_action('wp_ajax_nopriv_request_emergency_access', array(__CLASS__, 'ajax_request_emergency'));
    }

    /**
     * Check if emergency access URL parameter is present
     */
    public static function check_emergency_access() {
        // Check for emergency token in URL
        if (isset($_GET['fortress-emergency']) && !empty($_GET['fortress-emergency'])) {
            $provided_token = sanitize_text_field($_GET['fortress-emergency']);
            
            if (self::validate_emergency_token($provided_token)) {
                // Valid token - clear all lockouts and blacklist for this IP
                $ip = ArtistPro_IP_Manager::get_client_ip();
                
                // Clear lockout
                ArtistPro_Security_Core::clear_lockout($ip);
                
                // Remove from blacklist (temporary removal - 1 hour)
                self::temporary_whitelist($ip);
                
                // Log emergency access
                error_log("WP Fortress: Emergency access used from IP {$ip}");
                
                // Set cookie to prevent repeated use
                setcookie('fortress_emergency_used', '1', time() + 3600, COOKIEPATH, COOKIE_DOMAIN, is_ssl());
                
                // Show success message
                add_action('admin_notices', function() use ($ip) {
                    echo '<div class="notice notice-warning is-dismissible">';
                    echo '<p><strong>Emergency Access Granted!</strong> Your IP (' . esc_html($ip) . ') has been temporarily whitelisted for 1 hour.</p>';
                    echo '<p>Please change your password and review security settings immediately.</p>';
                    echo '</div>';
                });
                
                // Redirect to admin
                if (!is_admin()) {
                    wp_safe_redirect(admin_url());
                    exit;
                }
            } else {
                // Invalid token
                wp_die(
                    '<h1>Invalid Emergency Access Token</h1>' .
                    '<p>The emergency access token is invalid or has expired.</p>' .
                    '<p><a href="' . wp_login_url() . '">Return to Login</a></p>',
                    'Invalid Token',
                    array('response' => 403)
                );
            }
        }
    }

    /**
     * Validate emergency access token
     *
     * @param string $token Token to validate
     * @return bool
     */
    private static function validate_emergency_token($token) {
        $saved_token = get_option('artistpro_security_emergency_token');
        $token_expiry = get_option('artistpro_security_emergency_token_expiry');
        
        if (empty($saved_token) || empty($token_expiry)) {
            return false;
        }
        
        // Check if expired (tokens last 7 days)
        if (time() > $token_expiry) {
            return false;
        }
        
        // Constant-time comparison
        return hash_equals($saved_token, hash('sha256', $token));
    }

    /**
     * Generate new emergency access token
     *
     * @return string URL with emergency token
     */
    public static function generate_emergency_token() {
        // Generate cryptographically secure token
        $raw_token = bin2hex(random_bytes(32));
        $hashed_token = hash('sha256', $raw_token);
        
        // Save hashed token
        update_option('artistpro_security_emergency_token', $hashed_token);
        update_option('artistpro_security_emergency_token_expiry', time() + (7 * DAY_IN_SECONDS));
        update_option('artistpro_security_emergency_token_created', current_time('mysql'));
        
        // Return URL with raw token
        return add_query_arg('fortress-emergency', $raw_token, wp_login_url());
    }

    /**
     * Temporarily whitelist an IP for emergency access
     *
     * @param string $ip IP address
     */
    private static function temporary_whitelist($ip) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_IP_LISTS;
        
        // Remove from blacklist if exists
        $wpdb->delete(
            $table,
            array(
                'ip_address' => $ip,
                'list_type' => 'blacklist'
            ),
            array('%s', '%s')
        );
        
        // Add to temporary whitelist (1 hour)
        $wpdb->insert(
            $table,
            array(
                'ip_address' => $ip,
                'list_type' => 'whitelist',
                'source' => 'emergency_access',
                'reason' => 'Emergency access granted',
                'expires_at' => date('Y-m-d H:i:s', time() + 3600),
            ),
            array('%s', '%s', '%s', '%s', '%s')
        );
    }

    /**
     * Show emergency access notice to admins
     */
    public static function show_emergency_notice() {
        // Only show to administrators
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Only show on plugin settings page
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'artistpro-security') === false) {
            return;
        }
        
        // Check if emergency token exists
        $token_exists = get_option('artistpro_security_emergency_token');
        $token_created = get_option('artistpro_security_emergency_token_created');
        
        if (!$token_exists) {
            echo '<div class="notice notice-info">';
            echo '<p><strong>Tip:</strong> Generate an emergency access URL in case you lock yourself out. ';
            echo '<a href="#" id="generate-emergency-token" class="button button-small">Generate Now</a></p>';
            echo '</div>';
        } else {
            echo '<div class="notice notice-success is-dismissible">';
            echo '<p><strong>Emergency Access Configured</strong> - Created on ' . esc_html($token_created) . '. ';
            echo '<a href="#" id="regenerate-emergency-token" class="button button-small">Regenerate</a></p>';
            echo '</div>';
        }
    }

    /**
     * AJAX: Generate emergency token
     */
    public static function ajax_generate_token() {
        check_ajax_referer('artistpro_security_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $emergency_url = self::generate_emergency_token();
        
        wp_send_json_success(array(
            'url' => $emergency_url,
            'message' => 'Emergency access URL generated. Save this URL in a secure location!',
        ));
    }

    /**
     * AJAX: Request emergency access via email (no login required)
     */
    public static function ajax_request_emergency() {
        $email = isset($_POST['email']) ? sanitize_email($_POST['email']) : '';
        
        if (empty($email)) {
            wp_send_json_error('Email address required');
        }
        
        // Rate limit requests (1 per IP per hour)
        $ip = ArtistPro_IP_Manager::get_client_ip();
        $transient_key = 'emergency_request_' . md5($ip);
        
        if (get_transient($transient_key)) {
            wp_send_json_error('Please wait before requesting another emergency access link');
        }
        
        // Check if email belongs to an admin user
        $user = get_user_by('email', $email);
        if (!$user || !user_can($user, 'manage_options')) {
            // Don't reveal if user exists - always show success
            wp_send_json_success(array(
                'message' => 'If this email belongs to an administrator, an emergency access link has been sent.',
            ));
        }
        
        // Generate one-time token
        $emergency_url = self::generate_emergency_token();
        
        // Send email
        $subject = '[' . get_bloginfo('name') . '] Emergency Access Request';
        $message = "You requested emergency access to your WordPress site.\n\n";
        $message .= "Click the link below to regain access:\n";
        $message .= $emergency_url . "\n\n";
        $message .= "This link will expire in 7 days.\n\n";
        $message .= "If you did not request this, please ignore this email.\n\n";
        $message .= "IP Address: " . $ip . "\n";
        $message .= "Time: " . current_time('mysql') . "\n";
        
        wp_mail($email, $subject, $message);
        
        // Set rate limit
        set_transient($transient_key, '1', HOUR_IN_SECONDS);
        
        // Log request
        error_log("WP Fortress: Emergency access requested for {$email} from IP {$ip}");
        
        wp_send_json_success(array(
            'message' => 'If this email belongs to an administrator, an emergency access link has been sent.',
        ));
    }

    /**
     * Register settings
     */
    public static function register_settings() {
        register_setting('artistpro_security_settings', 'artistpro_security_emergency_enabled');
        register_setting('artistpro_security_settings', 'artistpro_security_emergency_token');
        register_setting('artistpro_security_settings', 'artistpro_security_emergency_token_expiry');
    }

    /**
     * Get emergency access instructions for SSH/FTP
     *
     * @return string Instructions
     */
    public static function get_ssh_instructions() {
        return <<<INSTRUCTIONS
=== EMERGENCY ACCESS VIA SSH/FTP ===

If you have SSH or FTP access, you can bypass the security plugin:

METHOD 1: Clear Your Lockout (Database)
----------------------------------------
mysql -u USERNAME -p DATABASE_NAME
DELETE FROM wp_artistpro_security_lockouts WHERE ip_address = 'YOUR_IP';
DELETE FROM wp_artistpro_security_ip_lists WHERE ip_address = 'YOUR_IP' AND list_type = 'blacklist';

METHOD 2: Temporarily Disable Plugin
-------------------------------------
Via SSH:
cd wp-content/plugins/
mv artistpro-security artistpro-security-disabled

Via FTP:
Rename the folder: artistpro-security → artistpro-security-disabled

METHOD 3: Add Your IP to Whitelist
-----------------------------------
mysql -u USERNAME -p DATABASE_NAME
INSERT INTO wp_artistpro_security_ip_lists (ip_address, list_type, source, reason)
VALUES ('YOUR_IP', 'whitelist', 'manual', 'Emergency access');

After regaining access, re-enable the plugin and review your settings.
INSTRUCTIONS;
    }
}
