<?php
/**
 * Dashboard Widget - Security overview
 *
 * @package ArtistPro_Security
 * @since 1.1.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_Dashboard_Widget {

    /**
     * Initialize dashboard widget
     */
    public static function init() {
        add_action('wp_dashboard_setup', array(__CLASS__, 'add_dashboard_widget'));
    }

    /**
     * Add dashboard widget
     */
    public static function add_dashboard_widget() {
        wp_add_dashboard_widget(
            'artistpro_security_dashboard',
            '🛡️ ArtistPro Security',
            array(__CLASS__, 'render_widget')
        );
    }

    /**
     * Render dashboard widget
     */
    public static function render_widget() {
        // Get 24-hour summary
        $summary = ArtistPro_Login_Logger::get_summary(24);
        $active_lockouts = ArtistPro_Security_Core::get_active_lockouts();
        $top_ips = ArtistPro_Login_Logger::get_top_ips(5, 24);
        
        // Calculate security score
        $score = self::calculate_security_score();
        
        ?>
        <div class="artistpro-security-widget">
            <div class="security-score" style="text-align: center; padding: 20px 0; border-bottom: 1px solid #ddd;">
                <div style="font-size: 48px; font-weight: bold; color: <?php echo self::get_score_color($score); ?>;">
                    <?php echo $score; ?>
                </div>
                <div style="font-size: 14px; color: #666; margin-top: 5px;">
                    Security Score
                </div>
                <div style="font-size: 12px; color: #999; margin-top: 5px;">
                    <?php echo self::get_score_label($score); ?>
                </div>
            </div>

            <div class="security-stats" style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px; padding: 15px 0;">
                <div style="text-align: center;">
                    <div style="font-size: 24px; font-weight: bold; color: #d63638;">
                        <?php echo number_format($summary['failed_attempts']); ?>
                    </div>
                    <div style="font-size: 12px; color: #666;">Failed Attempts (24h)</div>
                </div>
                
                <div style="text-align: center;">
                    <div style="font-size: 24px; font-weight: bold; color: #00a32a;">
                        <?php echo number_format($summary['blocked_attempts']); ?>
                    </div>
                    <div style="font-size: 12px; color: #666;">Blocked (24h)</div>
                </div>
                
                <div style="text-align: center;">
                    <div style="font-size: 24px; font-weight: bold; color: #2271b1;">
                        <?php echo number_format($summary['unique_ips']); ?>
                    </div>
                    <div style="font-size: 12px; color: #666;">Unique IPs</div>
                </div>
                
                <div style="text-align: center;">
                    <div style="font-size: 24px; font-weight: bold; color: <?php echo count($active_lockouts) > 0 ? '#d63638' : '#666'; ?>;">
                        <?php echo count($active_lockouts); ?>
                    </div>
                    <div style="font-size: 12px; color: #666;">Active Lockouts</div>
                </div>
            </div>

            <?php if (!empty($top_ips)): ?>
            <div style="border-top: 1px solid #ddd; padding-top: 15px;">
                <h4 style="margin: 0 0 10px 0; font-size: 13px;">🔥 Top Offending IPs (24h)</h4>
                <table style="width: 100%; font-size: 12px;">
                    <?php foreach ($top_ips as $ip_data): ?>
                    <tr>
                        <td style="padding: 5px 0;">
                            <code><?php echo esc_html($ip_data->ip_address); ?></code>
                            <?php if ($ip_data->country_code): ?>
                                <span style="font-size: 10px; color: #999;">
                                    <?php echo esc_html($ip_data->country_code); ?>
                                </span>
                            <?php endif; ?>
                        </td>
                        <td style="text-align: right; padding: 5px 0;">
                            <strong><?php echo number_format($ip_data->attempt_count); ?></strong> attempts
                        </td>
                        <td style="text-align: right; padding: 5px 0;">
                            <a href="#" class="button button-small" 
                               onclick="artistproSecurityBlacklistIP('<?php echo esc_js($ip_data->ip_address); ?>'); return false;">
                                Block
                            </a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </table>
            </div>
            <?php endif; ?>

            <div style="border-top: 1px solid #ddd; padding-top: 15px; margin-top: 15px;">
                <a href="<?php echo admin_url('options-general.php?page=artistpro-security'); ?>" 
                   class="button button-primary" style="width: 100%; text-align: center;">
                    View Full Security Dashboard
                </a>
            </div>
        </div>

        <script>
        function artistproSecurityBlacklistIP(ip) {
            if (!confirm('Blacklist IP ' + ip + '?')) {
                return;
            }
            
            jQuery.post(ajaxurl, {
                action: 'artistpro_blacklist_ip',
                ip: ip,
                nonce: '<?php echo wp_create_nonce('artistpro_security_admin'); ?>'
            }, function(response) {
                if (response.success) {
                    alert('IP blacklisted successfully!');
                    location.reload();
                } else {
                    alert('Error: ' + response.data);
                }
            });
        }
        </script>

        <style>
        .artistpro-security-widget {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif;
        }
        .artistpro-security-widget table {
            border-collapse: collapse;
        }
        .artistpro-security-widget tr:hover {
            background: #f9f9f9;
        }
        </style>
        <?php
    }

    /**
     * Calculate security score (0-100)
     *
     * @return int
     */
    private static function calculate_security_score() {
        $score = 100;
        
        // Check if 2FA is enabled for admins
        $admins = get_users(array('role' => 'administrator'));
        $admins_with_2fa = 0;
        foreach ($admins as $admin) {
            if (ArtistPro_Two_Factor_Auth::is_enabled_for_user($admin->ID)) {
                $admins_with_2fa++;
            }
        }
        $admin_2fa_percentage = count($admins) > 0 ? ($admins_with_2fa / count($admins)) * 100 : 0;
        if ($admin_2fa_percentage < 100) {
            $score -= (100 - $admin_2fa_percentage) * 0.3; // Up to -30 points
        }
        
        // Check for recent attacks
        $summary = ArtistPro_Login_Logger::get_summary(24);
        $attack_ratio = $summary['total_attempts'] > 0 
            ? ($summary['blocked_attempts'] / $summary['total_attempts']) * 100 
            : 100;
        
        if ($attack_ratio < 90) {
            $score -= (90 - $attack_ratio) * 0.5; // Up to -45 points
        }
        
        // Bonus for Cloudflare integration
        if (ArtistPro_Cloudflare_Integration::is_enabled()) {
            $score += 10;
        }
        
        // Bonus for reCAPTCHA
        if (ArtistPro_reCAPTCHA::is_enabled()) {
            $score += 5;
        }
        
        return max(0, min(100, round($score)));
    }

    /**
     * Get color for security score
     *
     * @param int $score Score (0-100)
     * @return string Color hex
     */
    private static function get_score_color($score) {
        if ($score >= 90) return '#00a32a'; // Green
        if ($score >= 70) return '#dba617'; // Yellow
        if ($score >= 50) return '#f0a800'; // Orange
        return '#d63638'; // Red
    }

    /**
     * Get label for security score
     *
     * @param int $score Score (0-100)
     * @return string Label
     */
    private static function get_score_label($score) {
        if ($score >= 90) return 'Excellent - Your site is well protected';
        if ($score >= 70) return 'Good - Consider enabling 2FA for all admins';
        if ($score >= 50) return 'Fair - Security improvements recommended';
        return 'Poor - Immediate action required';
    }
}
