<?php
/**
 * Cloudflare Integration - Sync blacklist to CF firewall
 * MAJOR SELLING POINT: Block attacks at edge, not at WordPress
 *
 * @package ArtistPro_Security
 * @since 1.1.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_Cloudflare_Integration {

    /**
     * Cloudflare API endpoint
     */
    const API_BASE = 'https://api.cloudflare.com/client/v4';

    /**
     * Initialize Cloudflare integration
     */
    public static function init() {
        // Only initialize if Cloudflare is enabled
        if (!self::is_enabled()) {
            return;
        }

        // Auto-sync when IP is blacklisted
        add_action('artistpro_security_ip_blacklisted', array(__CLASS__, 'sync_ip_to_cloudflare'), 10, 2);
        
        // Auto-remove when IP is removed from blacklist
        add_action('artistpro_security_ip_removed', array(__CLASS__, 'remove_ip_from_cloudflare'), 10, 1);
        
        // Check threat score on login attempts
        add_filter('artistpro_security_check_ip', array(__CLASS__, 'check_threat_score'), 10, 2);
        
        // Add settings
        add_action('admin_init', array(__CLASS__, 'register_settings'));
        
        // AJAX handlers
        add_action('wp_ajax_test_cloudflare_connection', array(__CLASS__, 'ajax_test_connection'));
        add_action('wp_ajax_sync_all_to_cloudflare', array(__CLASS__, 'ajax_sync_all'));
        add_action('wp_ajax_toggle_under_attack', array(__CLASS__, 'ajax_toggle_under_attack'));
        add_action('wp_ajax_get_cloudflare_stats', array(__CLASS__, 'ajax_get_stats'));
    }

    /**
     * Check if Cloudflare integration is enabled
     *
     * @return bool
     */
    public static function is_enabled() {
        return (bool) get_option('artistpro_security_cloudflare_enabled', false);
    }

    /**
     * Get Cloudflare API token
     *
     * @return string
     */
    private static function get_api_token() {
        return get_option('artistpro_security_cloudflare_api_token', '');
    }

    /**
     * Get Cloudflare Zone ID
     *
     * @return string
     */
    private static function get_zone_id() {
        return get_option('artistpro_security_cloudflare_zone_id', '');
    }

    /**
     * Make API request to Cloudflare
     *
     * @param string $endpoint API endpoint
     * @param string $method HTTP method
     * @param array $data Request data
     * @return array|WP_Error
     */
    private static function api_request($endpoint, $method = 'GET', $data = array()) {
        $api_token = self::get_api_token();
        
        if (empty($api_token)) {
            return new WP_Error('no_token', 'Cloudflare API token not configured');
        }

        $url = self::API_BASE . $endpoint;
        
        $args = array(
            'method' => $method,
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_token,
                'Content-Type' => 'application/json',
            ),
            'timeout' => 15,
        );

        if (!empty($data) && in_array($method, array('POST', 'PUT', 'PATCH'))) {
            $args['body'] = json_encode($data);
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            return $response;
        }

        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);

        if (!$result['success']) {
            $error_msg = isset($result['errors'][0]['message']) 
                ? $result['errors'][0]['message'] 
                : 'Unknown API error';
            return new WP_Error('api_error', $error_msg);
        }

        return $result;
    }

    /**
     * Test Cloudflare API connection
     *
     * @return array|WP_Error
     */
    public static function test_connection() {
        // Verify token by getting user info
        $response = self::api_request('/user/tokens/verify');
        
        if (is_wp_error($response)) {
            return $response;
        }

        return array(
            'success' => true,
            'message' => 'Cloudflare connection successful!',
            'status' => $response['result']['status'],
        );
    }

    /**
     * Get available zones for the account
     *
     * @return array|WP_Error
     */
    public static function get_zones() {
        $response = self::api_request('/zones?per_page=50');
        
        if (is_wp_error($response)) {
            return $response;
        }

        $zones = array();
        foreach ($response['result'] as $zone) {
            $zones[] = array(
                'id' => $zone['id'],
                'name' => $zone['name'],
                'status' => $zone['status'],
            );
        }

        return $zones;
    }

    /**
     * Sync IP to Cloudflare firewall
     *
     * @param string $ip IP address
     * @param string $reason Reason for blocking
     * @return bool|WP_Error
     */
    public static function sync_ip_to_cloudflare($ip, $reason = '') {
        $zone_id = self::get_zone_id();
        
        if (empty($zone_id)) {
            return new WP_Error('no_zone', 'Cloudflare Zone ID not configured');
        }

        // Check if rule already exists
        $existing = self::get_firewall_rule_by_ip($ip);
        if (!is_wp_error($existing) && !empty($existing)) {
            return true; // Already exists
        }

        // Create firewall rule
        $data = array(
            'mode' => 'block',
            'configuration' => array(
                'target' => 'ip',
                'value' => $ip,
            ),
            'notes' => 'ArtistPro Security: ' . ($reason ?: 'Auto-blocked from WordPress'),
        );

        $response = self::api_request("/zones/{$zone_id}/firewall/access_rules/rules", 'POST', $data);

        if (is_wp_error($response)) {
            error_log('ArtistPro Security: Failed to sync IP to Cloudflare: ' . $response->get_error_message());
            return $response;
        }

        // Store Cloudflare rule ID for later removal
        self::store_cloudflare_rule_id($ip, $response['result']['id']);

        error_log("ArtistPro Security: Synced IP {$ip} to Cloudflare firewall");
        
        return true;
    }

    /**
     * Remove IP from Cloudflare firewall
     *
     * @param string $ip IP address
     * @return bool|WP_Error
     */
    public static function remove_ip_from_cloudflare($ip) {
        $zone_id = self::get_zone_id();
        $rule_id = self::get_cloudflare_rule_id($ip);
        
        if (empty($zone_id) || empty($rule_id)) {
            return false;
        }

        $response = self::api_request("/zones/{$zone_id}/firewall/access_rules/rules/{$rule_id}", 'DELETE');

        if (is_wp_error($response)) {
            error_log('ArtistPro Security: Failed to remove IP from Cloudflare: ' . $response->get_error_message());
            return $response;
        }

        // Remove stored rule ID
        self::delete_cloudflare_rule_id($ip);

        error_log("ArtistPro Security: Removed IP {$ip} from Cloudflare firewall");
        
        return true;
    }

    /**
     * Get firewall rule by IP
     *
     * @param string $ip IP address
     * @return array|WP_Error
     */
    private static function get_firewall_rule_by_ip($ip) {
        $zone_id = self::get_zone_id();
        
        $response = self::api_request("/zones/{$zone_id}/firewall/access_rules/rules?configuration.value={$ip}");

        if (is_wp_error($response)) {
            return $response;
        }

        return $response['result'];
    }

    /**
     * Store Cloudflare rule ID for an IP
     *
     * @param string $ip IP address
     * @param string $rule_id Cloudflare rule ID
     */
    private static function store_cloudflare_rule_id($ip, $rule_id) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_IP_LISTS;
        
        $wpdb->update(
            $table,
            array('cidr_notation' => $rule_id), // Store rule ID in unused field
            array(
                'ip_address' => $ip,
                'list_type' => 'blacklist',
            ),
            array('%s'),
            array('%s', '%s')
        );
    }

    /**
     * Get Cloudflare rule ID for an IP
     *
     * @param string $ip IP address
     * @return string|null
     */
    private static function get_cloudflare_rule_id($ip) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_IP_LISTS;
        
        return $wpdb->get_var($wpdb->prepare(
            "SELECT cidr_notation FROM {$table} WHERE ip_address = %s AND list_type = 'blacklist'",
            $ip
        ));
    }

    /**
     * Delete stored Cloudflare rule ID
     *
     * @param string $ip IP address
     */
    private static function delete_cloudflare_rule_id($ip) {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_IP_LISTS;
        
        $wpdb->update(
            $table,
            array('cidr_notation' => null),
            array(
                'ip_address' => $ip,
                'list_type' => 'blacklist',
            ),
            array('%s'),
            array('%s', '%s')
        );
    }

    /**
     * Sync all blacklisted IPs to Cloudflare
     *
     * @return array Results
     */
    public static function sync_all_blacklisted_ips() {
        global $wpdb;
        $table = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_IP_LISTS;
        
        $blacklisted = $wpdb->get_results(
            "SELECT ip_address, reason FROM {$table} 
             WHERE list_type = 'blacklist' 
             AND (expires_at IS NULL OR expires_at > NOW())"
        );

        $results = array(
            'success' => 0,
            'failed' => 0,
            'skipped' => 0,
        );

        foreach ($blacklisted as $entry) {
            $result = self::sync_ip_to_cloudflare($entry->ip_address, $entry->reason);
            
            if (is_wp_error($result)) {
                $results['failed']++;
            } elseif ($result === true) {
                $results['success']++;
            } else {
                $results['skipped']++;
            }
        }

        return $results;
    }

    /**
     * Get IP threat score from Cloudflare
     *
     * @param string $ip IP address
     * @return int|WP_Error Threat score (0-100) or error
     */
    public static function get_threat_score($ip) {
        // Use Cloudflare's IP intelligence API
        $response = self::api_request("/accounts/" . self::get_account_id() . "/intel/ip?ipv4={$ip}");

        if (is_wp_error($response)) {
            return $response;
        }

        // Extract threat score
        return isset($response['result'][0]['risk']) ? (int) $response['result'][0]['risk'] : 0;
    }

    /**
     * Check threat score and auto-block if high
     *
     * @param bool $allow Current allow status
     * @param string $ip IP address
     * @return bool
     */
    public static function check_threat_score($allow, $ip) {
        // Only check if enabled
        if (!get_option('artistpro_security_cloudflare_threat_check', false)) {
            return $allow;
        }

        $threshold = (int) get_option('artistpro_security_cloudflare_threat_threshold', 80);
        
        $score = self::get_threat_score($ip);
        
        if (!is_wp_error($score) && $score >= $threshold) {
            // High threat score - add to blacklist
            ArtistPro_IP_Manager::add_to_blacklist(
                $ip,
                "High Cloudflare threat score: {$score}",
                'cloudflare_threat',
                null
            );
            
            return false;
        }

        return $allow;
    }

    /**
     * Toggle "Under Attack" mode
     *
     * @param bool $enable Enable or disable
     * @return bool|WP_Error
     */
    public static function toggle_under_attack_mode($enable = true) {
        $zone_id = self::get_zone_id();
        
        $data = array(
            'value' => $enable ? 'under_attack' : 'off',
        );

        $response = self::api_request("/zones/{$zone_id}/settings/security_level", 'PATCH', $data);

        if (is_wp_error($response)) {
            return $response;
        }

        update_option('artistpro_security_cloudflare_under_attack', $enable);
        
        return true;
    }

    /**
     * Get Cloudflare analytics/stats
     *
     * @return array|WP_Error
     */
    public static function get_analytics() {
        $zone_id = self::get_zone_id();
        
        // Get last 24 hours of data
        $since = date('c', strtotime('-24 hours'));
        $until = date('c');
        
        $response = self::api_request(
            "/zones/{$zone_id}/analytics/dashboard?since={$since}&until={$until}"
        );

        if (is_wp_error($response)) {
            return $response;
        }

        $data = $response['result'];
        
        return array(
            'requests' => array(
                'all' => $data['totals']['requests']['all'],
                'cached' => $data['totals']['requests']['cached'],
                'uncached' => $data['totals']['requests']['uncached'],
            ),
            'bandwidth' => array(
                'all' => $data['totals']['bandwidth']['all'],
                'cached' => $data['totals']['bandwidth']['cached'],
                'uncached' => $data['totals']['bandwidth']['uncached'],
            ),
            'threats' => array(
                'all' => $data['totals']['threats']['all'],
                'type' => $data['totals']['threats']['type'] ?? array(),
            ),
        );
    }

    /**
     * Get account ID (needed for some API calls)
     *
     * @return string
     */
    private static function get_account_id() {
        // Try to get from option first
        $account_id = get_option('artistpro_security_cloudflare_account_id');
        
        if (!empty($account_id)) {
            return $account_id;
        }

        // Fetch from API
        $response = self::api_request('/accounts');
        
        if (!is_wp_error($response) && !empty($response['result'])) {
            $account_id = $response['result'][0]['id'];
            update_option('artistpro_security_cloudflare_account_id', $account_id);
            return $account_id;
        }

        return '';
    }

    /**
     * Register settings
     */
    public static function register_settings() {
        register_setting('artistpro_security_cloudflare', 'artistpro_security_cloudflare_enabled');
        register_setting('artistpro_security_cloudflare', 'artistpro_security_cloudflare_api_token');
        register_setting('artistpro_security_cloudflare', 'artistpro_security_cloudflare_zone_id');
        register_setting('artistpro_security_cloudflare', 'artistpro_security_cloudflare_threat_check');
        register_setting('artistpro_security_cloudflare', 'artistpro_security_cloudflare_threat_threshold');
        register_setting('artistpro_security_cloudflare', 'artistpro_security_cloudflare_auto_sync');
    }

    /**
     * AJAX: Test connection
     */
    public static function ajax_test_connection() {
        check_ajax_referer('artistpro_security_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        $result = self::test_connection();
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Sync all blacklisted IPs
     */
    public static function ajax_sync_all() {
        check_ajax_referer('artistpro_security_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        $results = self::sync_all_blacklisted_ips();
        
        wp_send_json_success($results);
    }

    /**
     * AJAX: Toggle Under Attack mode
     */
    public static function ajax_toggle_under_attack() {
        check_ajax_referer('artistpro_security_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        $enable = isset($_POST['enable']) ? (bool) $_POST['enable'] : true;
        
        $result = self::toggle_under_attack_mode($enable);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }

        wp_send_json_success(array(
            'enabled' => $enable,
            'message' => $enable ? 'Under Attack mode enabled' : 'Under Attack mode disabled',
        ));
    }

    /**
     * AJAX: Get Cloudflare stats
     */
    public static function ajax_get_stats() {
        check_ajax_referer('artistpro_security_admin', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        $stats = self::get_analytics();
        
        if (is_wp_error($stats)) {
            wp_send_json_error($stats->get_error_message());
        }

        wp_send_json_success($stats);
    }
}
