<?php
/**
 * Blocklist Sync - Sync IP blocklists from FireHOL, Spamhaus, etc.
 *
 * @package ArtistPro_Security
 * @since 1.0.0
 */

if (!defined('WPINC')) {
    die;
}

class ArtistPro_Blocklist_Sync {

    /**
     * @var array Available blocklist sources
     */
    private $sources = [
        'firehol_level1' => [
            'name' => 'FireHOL Level 1',
            'url' => 'https://iplists.firehol.org/files/firehol_level1.netset',
            'description' => 'Aggregated blocklist - Spamhaus, dshield, feodotracker',
            'default_enabled' => true
        ],
        'spamhaus_drop' => [
            'name' => 'Spamhaus DROP',
            'url' => 'https://www.spamhaus.org/drop/drop.txt',
            'description' => 'Hijacked IP ranges - do not route',
            'default_enabled' => false
        ],
        'spamhaus_edrop' => [
            'name' => 'Spamhaus EDROP',
            'url' => 'https://www.spamhaus.org/drop/edrop.txt',
            'description' => 'Extended DROP list',
            'default_enabled' => false
        ],
        'blocklist_de' => [
            'name' => 'Blocklist.de All',
            'url' => 'https://lists.blocklist.de/lists/all.txt',
            'description' => 'Active attackers (48hr rolling)',
            'default_enabled' => false
        ]
    ];

    /**
     * Initialize hooks
     */
    public function init() {
        // Register cron hook
        add_action('artistpro_security_sync_blocklists', [$this, 'sync_all']);
    }

    /**
     * Get available blocklist sources
     *
     * @return array
     */
    public function get_sources() {
        return $this->sources;
    }

    /**
     * Get enabled blocklist sources
     *
     * @return array
     */
    public function get_enabled_sources() {
        $enabled = get_option('artistpro_security_blocklists_enabled', ['firehol_level1']);
        return array_intersect_key($this->sources, array_flip($enabled));
    }

    /**
     * Enable a blocklist source
     *
     * @param string $source_key
     * @return bool
     */
    public function enable_source($source_key) {
        if (!isset($this->sources[$source_key])) {
            return false;
        }

        $enabled = get_option('artistpro_security_blocklists_enabled', ['firehol_level1']);
        if (!in_array($source_key, $enabled)) {
            $enabled[] = $source_key;
            update_option('artistpro_security_blocklists_enabled', $enabled);
        }

        return true;
    }

    /**
     * Disable a blocklist source
     *
     * @param string $source_key
     * @return bool
     */
    public function disable_source($source_key) {
        $enabled = get_option('artistpro_security_blocklists_enabled', ['firehol_level1']);
        $enabled = array_diff($enabled, [$source_key]);
        update_option('artistpro_security_blocklists_enabled', array_values($enabled));

        // Remove IPs from this source
        $this->remove_source_ips($source_key);

        return true;
    }

    /**
     * Sync all enabled blocklists
     *
     * @return array Results for each source
     */
    public function sync_all() {
        $results = [];
        $enabled = $this->get_enabled_sources();

        foreach ($enabled as $key => $source) {
            $results[$key] = $this->sync_source($key);
        }

        update_option('artistpro_security_blocklist_last_sync', time());

        return $results;
    }

    /**
     * Sync a single blocklist source
     *
     * @param string $source_key
     * @return array ['success' => bool, 'count' => int, 'message' => string]
     */
    public function sync_source($source_key) {
        if (!isset($this->sources[$source_key])) {
            return ['success' => false, 'count' => 0, 'message' => 'Unknown source'];
        }

        $source = $this->sources[$source_key];
        $url = $source['url'];

        // Download the list
        $response = wp_remote_get($url, [
            'timeout' => 60,
            'user-agent' => 'ArtistPro-Security/1.0 (WordPress Security Plugin)'
        ]);

        if (is_wp_error($response)) {
            return [
                'success' => false,
                'count' => 0,
                'message' => 'Download failed: ' . $response->get_error_message()
            ];
        }

        $body = wp_remote_retrieve_body($response);
        if (empty($body)) {
            return ['success' => false, 'count' => 0, 'message' => 'Empty response'];
        }

        // Parse the list
        $ips = $this->parse_blocklist($body, $source_key);

        if (empty($ips)) {
            return ['success' => false, 'count' => 0, 'message' => 'No IPs parsed'];
        }

        // Update database
        $count = $this->update_blocklist_ips($source_key, $ips);

        // Update last sync time for this source
        $sync_times = get_option('artistpro_security_blocklist_sync_times', []);
        $sync_times[$source_key] = time();
        update_option('artistpro_security_blocklist_sync_times', $sync_times);

        return [
            'success' => true,
            'count' => $count,
            'message' => "Synced {$count} IP ranges"
        ];
    }

    /**
     * Parse blocklist content
     *
     * @param string $content Raw blocklist content
     * @param string $source_key Source identifier
     * @return array List of IPs/CIDRs
     */
    private function parse_blocklist($content, $source_key) {
        $ips = [];
        $lines = explode("\n", $content);

        foreach ($lines as $line) {
            $line = trim($line);

            // Skip empty lines and comments
            if (empty($line) || $line[0] === '#' || $line[0] === ';') {
                continue;
            }

            // Handle different formats
            if ($source_key === 'spamhaus_drop' || $source_key === 'spamhaus_edrop') {
                // Format: IP/CIDR ; SBL# (comment)
                $parts = explode(';', $line);
                $ip = trim($parts[0]);
            } else {
                // FireHOL and blocklist.de use plain IP/CIDR format
                $ip = $line;
            }

            // Validate IP or CIDR
            if ($this->is_valid_ip_or_cidr($ip)) {
                $ips[] = $ip;
            }
        }

        return $ips;
    }

    /**
     * Validate IP address or CIDR notation
     *
     * @param string $ip IP or CIDR
     * @return bool
     */
    private function is_valid_ip_or_cidr($ip) {
        // Check for CIDR notation
        if (strpos($ip, '/') !== false) {
            list($ip_part, $cidr) = explode('/', $ip, 2);
            if (!filter_var($ip_part, FILTER_VALIDATE_IP)) {
                return false;
            }
            $cidr = intval($cidr);
            // IPv4 CIDR: 0-32, IPv6 CIDR: 0-128
            if (filter_var($ip_part, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
                return $cidr >= 0 && $cidr <= 32;
            } else {
                return $cidr >= 0 && $cidr <= 128;
            }
        }

        return filter_var($ip, FILTER_VALIDATE_IP) !== false;
    }

    /**
     * Update blocklist IPs in database
     *
     * @param string $source_key Source identifier
     * @param array $ips List of IPs/CIDRs
     * @return int Number of IPs added/updated
     */
    private function update_blocklist_ips($source_key, $ips) {
        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_ip_lists';

        // Remove old IPs from this source
        $this->remove_source_ips($source_key);

        // Insert new IPs in batches
        $batch_size = 500;
        $count = 0;

        foreach (array_chunk($ips, $batch_size) as $batch) {
            $values = [];
            $placeholders = [];

            foreach ($batch as $ip) {
                $placeholders[] = "(%s, 'blacklist', %s, %s, NULL)";
                $values[] = $ip;
                $values[] = $source_key;
                $values[] = $this->sources[$source_key]['name'] ?? $source_key;
            }

            if (!empty($placeholders)) {
                $sql = "INSERT INTO {$table} (ip_address, list_type, source, reason, expires_at) VALUES "
                     . implode(', ', $placeholders);

                $wpdb->query($wpdb->prepare($sql, $values));
                $count += count($batch);
            }
        }

        return $count;
    }

    /**
     * Remove IPs from a specific source
     *
     * @param string $source_key Source identifier
     * @return int Number of rows deleted
     */
    private function remove_source_ips($source_key) {
        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_ip_lists';

        return $wpdb->delete($table, ['source' => $source_key], ['%s']);
    }

    /**
     * Get sync status for all sources
     *
     * @return array
     */
    public function get_sync_status() {
        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_ip_lists';

        $enabled = get_option('artistpro_security_blocklists_enabled', ['firehol_level1']);
        $sync_times = get_option('artistpro_security_blocklist_sync_times', []);

        $status = [];

        foreach ($this->sources as $key => $source) {
            // Get count for this source
            $count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$table} WHERE source = %s",
                $key
            ));

            $status[$key] = [
                'name' => $source['name'],
                'description' => $source['description'],
                'enabled' => in_array($key, $enabled),
                'ip_count' => intval($count),
                'last_sync' => $sync_times[$key] ?? null,
                'last_sync_human' => isset($sync_times[$key])
                    ? human_time_diff($sync_times[$key]) . ' ago'
                    : 'Never'
            ];
        }

        return $status;
    }

    /**
     * Get total blocklist IP count
     *
     * @return int
     */
    public function get_total_count() {
        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_ip_lists';

        $enabled = get_option('artistpro_security_blocklists_enabled', ['firehol_level1']);

        if (empty($enabled)) {
            return 0;
        }

        $placeholders = implode(',', array_fill(0, count($enabled), '%s'));

        return intval($wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE source IN ({$placeholders})",
            $enabled
        )));
    }

    /**
     * Check if IP is in any blocklist
     *
     * @param string $ip IP address to check
     * @return array|null Source info if blocked, null if not
     */
    public function check_ip($ip) {
        global $wpdb;
        $table = $wpdb->prefix . 'artistpro_security_ip_lists';

        // First check exact match
        $match = $wpdb->get_row($wpdb->prepare(
            "SELECT ip_address, source, reason FROM {$table}
             WHERE ip_address = %s AND list_type = 'blacklist'",
            $ip
        ));

        if ($match) {
            return [
                'blocked' => true,
                'ip' => $match->ip_address,
                'source' => $match->source,
                'reason' => $match->reason
            ];
        }

        // Check CIDR ranges (IPv4 only for performance)
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            $ip_long = ip2long($ip);

            // Get all CIDR ranges
            $ranges = $wpdb->get_results(
                "SELECT ip_address, source, reason FROM {$table}
                 WHERE ip_address LIKE '%/%' AND list_type = 'blacklist'"
            );

            foreach ($ranges as $range) {
                if ($this->ip_in_cidr($ip, $range->ip_address)) {
                    return [
                        'blocked' => true,
                        'ip' => $range->ip_address,
                        'source' => $range->source,
                        'reason' => $range->reason
                    ];
                }
            }
        }

        return null;
    }

    /**
     * Check if IP is in CIDR range
     *
     * @param string $ip IP address
     * @param string $cidr CIDR notation (e.g., 192.168.1.0/24)
     * @return bool
     */
    private function ip_in_cidr($ip, $cidr) {
        if (strpos($cidr, '/') === false) {
            return $ip === $cidr;
        }

        list($subnet, $bits) = explode('/', $cidr);

        $ip_long = ip2long($ip);
        $subnet_long = ip2long($subnet);
        $mask = -1 << (32 - intval($bits));

        $subnet_long &= $mask;

        return ($ip_long & $mask) === $subnet_long;
    }

    /**
     * Force sync now (manual trigger)
     *
     * @return array Results
     */
    public function force_sync() {
        return $this->sync_all();
    }
}
