/**
 * artistpro Security - Admin JavaScript
 */

(function($) {
    'use strict';

    const Security = {
        nonce: artistproSecurity.nonce,

        init: function() {
            this.bindEvents();
        },

        bindEvents: function() {
            // IP Management
            $(document).on('click', '.block-ip', this.blockIP.bind(this));
            $(document).on('click', '.whitelist-ip', this.whitelistIP.bind(this));
            $(document).on('click', '.remove-ip', this.removeIP.bind(this));
            $(document).on('click', '.unlock-ip', this.unlockIP.bind(this));
            $(document).on('click', '.whitelist-self', this.whitelistIP.bind(this));
            $(document).on('submit', '#add-ip-form', this.addIP.bind(this));

            // Country Blocking
            $(document).on('change', '.country-checkbox', this.toggleCountry.bind(this));
            $(document).on('click', '.block-high-risk', this.blockHighRisk.bind(this));
            $(document).on('click', '.unblock-all', this.unblockAll.bind(this));

            // Blocklist Management
            $(document).on('change', '.toggle-blocklist', this.toggleBlocklist.bind(this));
            $(document).on('click', '.sync-blocklist', this.syncBlocklist.bind(this));
            $(document).on('click', '.sync-all-blocklists', this.syncAllBlocklists.bind(this));

            // Logs
            $(document).on('click', '.clear-logs', this.showClearLogsModal.bind(this));
            $(document).on('click', '.confirm-clear-logs', this.clearLogs.bind(this));
            $(document).on('click', '.export-logs', this.exportLogs.bind(this));
            $(document).on('click', '.cancel-modal', this.hideModals.bind(this));

            // Settings
            $(document).on('submit', '#security-settings-form', this.saveSettings.bind(this));
            $(document).on('click', '.download-geoip', this.downloadGeoIP.bind(this));
        },

        ajax: function(action, data, callback) {
            data.action = 'artistpro_security_' + action;
            data.nonce = this.nonce;

            $.post(ajaxurl, data, function(response) {
                if (callback) callback(response);
            });
        },

        blockIP: function(e) {
            e.preventDefault();
            const ip = $(e.currentTarget).data('ip');
            const reason = prompt('Reason for blocking (optional):', 'Suspicious activity');

            if (reason === null) return; // Cancelled

            this.ajax('add_ip', {
                ip: ip,
                list_type: 'blacklist',
                reason: reason
            }, function(response) {
                if (response.success) {
                    alert('IP blocked successfully');
                    location.reload();
                } else {
                    alert('Error: ' + response.data.message);
                }
            });
        },

        whitelistIP: function(e) {
            e.preventDefault();
            const ip = $(e.currentTarget).data('ip');
            const reason = prompt('Reason for whitelisting (optional):', 'Trusted IP');

            if (reason === null) return;

            this.ajax('add_ip', {
                ip: ip,
                list_type: 'whitelist',
                reason: reason
            }, function(response) {
                if (response.success) {
                    alert('IP whitelisted successfully');
                    location.reload();
                } else {
                    alert('Error: ' + response.data.message);
                }
            });
        },

        removeIP: function(e) {
            e.preventDefault();
            const ip = $(e.currentTarget).data('ip');

            if (!confirm('Remove ' + ip + ' from the list?')) return;

            this.ajax('remove_ip', { ip: ip }, function(response) {
                if (response.success) {
                    $(e.currentTarget).closest('tr').fadeOut();
                } else {
                    alert('Error: ' + response.data.message);
                }
            });
        },

        unlockIP: function(e) {
            e.preventDefault();
            const ip = $(e.currentTarget).data('ip');

            if (!confirm('Unlock ' + ip + '?')) return;

            this.ajax('unlock_ip', { ip: ip }, function(response) {
                if (response.success) {
                    $(e.currentTarget).closest('tr').fadeOut();
                } else {
                    alert('Error: ' + response.data.message);
                }
            });
        },

        addIP: function(e) {
            e.preventDefault();
            const $form = $(e.currentTarget);
            const data = {
                ip: $form.find('[name="ip"]').val(),
                list_type: $form.find('[name="list_type"]').val(),
                reason: $form.find('[name="reason"]').val(),
                expires: $form.find('[name="expires"]').val()
            };

            this.ajax('add_ip', data, function(response) {
                if (response.success) {
                    alert('IP added successfully');
                    location.reload();
                } else {
                    alert('Error: ' + response.data.message);
                }
            });
        },

        toggleCountry: function(e) {
            const $checkbox = $(e.currentTarget);
            const code = $checkbox.data('code');
            const blocked = $checkbox.is(':checked');

            const action = blocked ? 'block_country' : 'unblock_country';

            this.ajax(action, { country_code: code }, function(response) {
                if (response.success) {
                    $checkbox.closest('.country-item').toggleClass('blocked', blocked);
                } else {
                    $checkbox.prop('checked', !blocked);
                    alert('Error: ' + response.data.message);
                }
            });
        },

        blockHighRisk: function(e) {
            e.preventDefault();
            const countries = $(e.currentTarget).data('countries').split(',');

            if (!confirm('Block ' + countries.length + ' high-risk countries?')) return;

            const self = this;
            countries.forEach(function(code) {
                self.ajax('block_country', { country_code: code });
                $('[data-code="' + code + '"]').prop('checked', true).closest('.country-item').addClass('blocked');
            });

            alert('High-risk countries blocked');
        },

        unblockAll: function(e) {
            e.preventDefault();

            if (!confirm('Unblock ALL countries?')) return;

            const self = this;
            $('.country-checkbox:checked').each(function() {
                const code = $(this).data('code');
                self.ajax('unblock_country', { country_code: code });
            });

            $('.country-checkbox').prop('checked', false);
            $('.country-item').removeClass('blocked');
            alert('All countries unblocked');
        },

        toggleBlocklist: function(e) {
            const $toggle = $(e.currentTarget);
            const source = $toggle.data('source');
            const enabled = $toggle.is(':checked');

            const $row = $toggle.closest('tr');
            const $syncBtn = $row.find('.sync-blocklist');

            $toggle.prop('disabled', true);

            this.ajax('toggle_blocklist', {
                source: source,
                enabled: enabled ? 1 : 0
            }, function(response) {
                $toggle.prop('disabled', false);
                $syncBtn.prop('disabled', !enabled);

                if (response.success && enabled && response.data.sync_result) {
                    $row.find('td:eq(2)').text(response.data.sync_result.count.toLocaleString());
                }
            });
        },

        syncBlocklist: function(e) {
            e.preventDefault();
            const $btn = $(e.currentTarget);
            const source = $btn.data('source');
            const originalText = $btn.text();

            $btn.text('Syncing...').prop('disabled', true);

            this.ajax('sync_blocklist', { source: source }, function(response) {
                $btn.text(originalText).prop('disabled', false);

                if (response.success) {
                    alert('Synced ' + response.data.count + ' IPs');
                    location.reload();
                } else {
                    alert('Sync failed: ' + response.data.message);
                }
            });
        },

        syncAllBlocklists: function(e) {
            e.preventDefault();
            const $btn = $(e.currentTarget);
            const originalText = $btn.text();

            $btn.text('Syncing...').prop('disabled', true);

            this.ajax('sync_blocklist', {}, function(response) {
                $btn.text(originalText).prop('disabled', false);

                if (response.success) {
                    let total = 0;
                    Object.values(response.data).forEach(r => {
                        if (r.success) total += r.count;
                    });
                    alert('Synced ' + total.toLocaleString() + ' IPs total');
                    location.reload();
                }
            });
        },

        showClearLogsModal: function(e) {
            e.preventDefault();
            $('#clear-logs-modal').show();
        },

        hideModals: function() {
            $('.modal').hide();
        },

        clearLogs: function(e) {
            e.preventDefault();
            const days = $('#clear-logs-days').val();

            if (!confirm('Are you sure? This cannot be undone.')) return;

            this.ajax('clear_logs', { days: days }, function(response) {
                if (response.success) {
                    alert('Logs cleared');
                    location.reload();
                }
            });
        },

        exportLogs: function(e) {
            e.preventDefault();
            const $btn = $(e.currentTarget);
            const originalText = $btn.text();

            $btn.text('Exporting...').prop('disabled', true);

            this.ajax('export_logs', {}, function(response) {
                $btn.text(originalText).prop('disabled', false);

                if (response.success) {
                    // Download the CSV
                    const blob = new Blob([atob(response.data.content)], { type: 'text/csv' });
                    const url = window.URL.createObjectURL(blob);
                    const a = document.createElement('a');
                    a.href = url;
                    a.download = response.data.filename;
                    a.click();
                    window.URL.revokeObjectURL(url);
                } else {
                    alert('Export failed: ' + response.data.message);
                }
            });
        },

        saveSettings: function(e) {
            e.preventDefault();
            const $form = $(e.currentTarget);
            const $btn = $form.find('button[type="submit"]');
            const $spinner = $form.find('.spinner');
            const $status = $form.find('.save-status');

            const data = {};
            $form.serializeArray().forEach(function(item) {
                data[item.name] = item.value;
            });

            // Handle unchecked checkboxes
            $form.find('input[type="checkbox"]').each(function() {
                if (!$(this).is(':checked')) {
                    data[$(this).attr('name')] = 0;
                }
            });

            $btn.prop('disabled', true);
            $spinner.addClass('is-active');

            this.ajax('save_settings', data, function(response) {
                $btn.prop('disabled', false);
                $spinner.removeClass('is-active');

                if (response.success) {
                    $status.text('Saved!').css('color', 'green');
                    setTimeout(() => $status.text(''), 2000);
                } else {
                    $status.text('Error: ' + response.data.message).css('color', 'red');
                }
            });
        },

        downloadGeoIP: function(e) {
            e.preventDefault();
            const $btn = $(e.currentTarget);

            $btn.text('Downloading...').prop('disabled', true);

            // Trigger download via AJAX
            $.post(ajaxurl, {
                action: 'artistpro_security_download_geoip',
                nonce: this.nonce
            }, function(response) {
                $btn.text('Download/Update GeoIP Database').prop('disabled', false);

                if (response.success) {
                    alert('GeoIP database updated!');
                    location.reload();
                } else {
                    alert('Download failed. Check your MaxMind license key.');
                }
            });
        }
    };

    $(document).ready(function() {
        Security.init();
    });

})(jQuery);
