<?php
/**
 * Plugin Name: artistpro Security
 * Plugin URI: https://artistpro.media
 * Description: Login security, IP rate limiting, blocklist integration, and country blocking for WordPress and the artistpro portal.
 * Version: 1.1.0
 * Author: artistpro Media
 * Author URI: https://artistpro.media
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: artistpro-security
 *
 * @package ArtistPro_Security
 * @since 1.0.0
 */

if (!defined('WPINC')) {
    die;
}

// Plugin constants
define('ARTISTPRO_SECURITY_VERSION', '1.1.0');
define('ARTISTPRO_SECURITY_DIR', plugin_dir_path(__FILE__));
define('ARTISTPRO_SECURITY_PATH', plugin_dir_path(__FILE__));  // Alias for consistency
define('ARTISTPRO_SECURITY_URL', plugin_dir_url(__FILE__));
define('ARTISTPRO_SECURITY_BASENAME', plugin_basename(__FILE__));

// Database table names (without prefix)
define('ARTISTPRO_SECURITY_TABLE_LOGS', 'artistpro_security_logs');
define('ARTISTPRO_SECURITY_TABLE_LOCKOUTS', 'artistpro_security_lockouts');
define('ARTISTPRO_SECURITY_TABLE_IP_LISTS', 'artistpro_security_ip_lists');
define('ARTISTPRO_SECURITY_TABLE_BLOCKED_COUNTRIES', 'artistpro_security_blocked_countries');
define('ARTISTPRO_SECURITY_TABLE_STATS', 'artistpro_security_stats');

// Load required classes

// Load required classes
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-ip-manager.php';
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-login-logger.php';
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-security-core.php';
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-geo-blocker.php';
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-blocklist-sync.php';
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-security-cron.php';
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-security-ajax.php';
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-security-notifier.php';
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-portal-integration.php';
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-cloudflare-integration.php';
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-emergency-access.php';
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-two-factor-auth.php';
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-recaptcha.php';
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-dashboard-widget.php';
require_once ARTISTPRO_SECURITY_DIR . 'includes/class-license-updater.php';
function artistpro_security_activate() {
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');

    // Login logs table
    $table_logs = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOGS;
    $sql_logs = "CREATE TABLE $table_logs (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        ip_address varchar(45) NOT NULL,
        username varchar(255) DEFAULT NULL,
        email varchar(255) DEFAULT NULL,
        login_type varchar(50) NOT NULL DEFAULT 'wp_login',
        result varchar(20) NOT NULL DEFAULT 'failed',
        failure_reason varchar(255) DEFAULT NULL,
        user_agent text,
        country_code char(2) DEFAULT NULL,
        city varchar(100) DEFAULT NULL,
        created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY ip_address (ip_address),
        KEY username (username),
        KEY created_at (created_at),
        KEY result (result),
        KEY login_type (login_type)
    ) $charset_collate;";
    dbDelta($sql_logs);

    // Active lockouts table
    $table_lockouts = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_LOCKOUTS;
    $sql_lockouts = "CREATE TABLE $table_lockouts (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        ip_address varchar(45) NOT NULL,
        username varchar(255) DEFAULT NULL,
        lockout_type varchar(20) NOT NULL DEFAULT 'short',
        attempts_count int unsigned NOT NULL DEFAULT 0,
        locked_until datetime NOT NULL,
        reason varchar(255) DEFAULT NULL,
        created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        UNIQUE KEY ip_address (ip_address),
        KEY locked_until (locked_until)
    ) $charset_collate;";
    dbDelta($sql_lockouts);

    // IP whitelist/blacklist table
    $table_ip_lists = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_IP_LISTS;
    $sql_ip_lists = "CREATE TABLE $table_ip_lists (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        ip_address varchar(45) NOT NULL,
        ip_range_start varchar(45) DEFAULT NULL,
        ip_range_end varchar(45) DEFAULT NULL,
        cidr_notation varchar(50) DEFAULT NULL,
        list_type varchar(20) NOT NULL DEFAULT 'blacklist',
        source varchar(50) NOT NULL DEFAULT 'manual',
        reason varchar(500) DEFAULT NULL,
        expires_at datetime DEFAULT NULL,
        created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
        updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY ip_address (ip_address),
        KEY list_type (list_type),
        KEY source (source),
        KEY expires_at (expires_at)
    ) $charset_collate;";
    dbDelta($sql_ip_lists);

    // Blocked countries table
    $table_countries = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_BLOCKED_COUNTRIES;
    $sql_countries = "CREATE TABLE $table_countries (
        id int unsigned NOT NULL AUTO_INCREMENT,
        country_code char(2) NOT NULL,
        country_name varchar(100) NOT NULL,
        block_type varchar(20) NOT NULL DEFAULT 'login_only',
        created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        UNIQUE KEY country_code (country_code)
    ) $charset_collate;";
    dbDelta($sql_countries);

    // Stats table (hourly aggregates)
    $table_stats = $wpdb->prefix . ARTISTPRO_SECURITY_TABLE_STATS;
    $sql_stats = "CREATE TABLE $table_stats (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        stat_date date NOT NULL,
        stat_hour tinyint unsigned DEFAULT NULL,
        total_attempts int unsigned NOT NULL DEFAULT 0,
        failed_attempts int unsigned NOT NULL DEFAULT 0,
        successful_logins int unsigned NOT NULL DEFAULT 0,
        blocked_attempts int unsigned NOT NULL DEFAULT 0,
        lockouts_triggered int unsigned NOT NULL DEFAULT 0,
        unique_ips int unsigned NOT NULL DEFAULT 0,
        PRIMARY KEY (id),
        UNIQUE KEY stat_period (stat_date, stat_hour)
    ) $charset_collate;";
    dbDelta($sql_stats);

    // Set default options
    $defaults = array(
        'artistpro_security_allowed_retries' => 4,
        'artistpro_security_lockout_duration' => 20,
        'artistpro_security_long_lockout_duration' => 24,
        'artistpro_security_lockouts_before_long' => 4,
        'artistpro_security_valid_duration' => 43200,
        'artistpro_security_notify_email' => get_option('admin_email'),
        'artistpro_security_notify_after_lockouts' => 3,
        'artistpro_security_daily_summary' => false,
        'artistpro_security_firehol_enabled' => true,
        'artistpro_security_spamhaus_enabled' => true,
        'artistpro_security_blocklist_de_enabled' => false,
        'artistpro_security_auto_blacklist_threshold' => 10,
        'artistpro_security_log_retention_days' => 30,
        'artistpro_security_trusted_proxies' => '',
        'artistpro_security_proxy_header' => 'HTTP_X_FORWARDED_FOR',
    );

    foreach ($defaults as $key => $value) {
        if (get_option($key) === false) {
            add_option($key, $value);
        }
    }

    // Schedule cron jobs
    if (!wp_next_scheduled('artistpro_security_sync_blocklists')) {
        wp_schedule_event(time(), 'daily', 'artistpro_security_sync_blocklists');
    }

    if (!wp_next_scheduled('artistpro_security_cleanup')) {
        wp_schedule_event(time(), 'daily', 'artistpro_security_cleanup');
    }

    // Trigger initial blocklist sync
    if (get_option('artistpro_security_firehol_enabled')) {
        wp_schedule_single_event(time() + 30, 'artistpro_security_initial_sync');
    }

    // Store activation time
    update_option('artistpro_security_activated', time());
}
register_activation_hook(__FILE__, 'artistpro_security_activate');

/**
 * Deactivation hook - clean up cron jobs
 */
function artistpro_security_deactivate() {
    wp_clear_scheduled_hook('artistpro_security_sync_blocklists');
    wp_clear_scheduled_hook('artistpro_security_cleanup');
    wp_clear_scheduled_hook('artistpro_security_initial_sync');
}
register_deactivation_hook(__FILE__, 'artistpro_security_deactivate');

/**
 * Global instances
 */
global $artistpro_security;
$artistpro_security = array();

/**
 * Initialize the plugin
 */
function artistpro_security_init() {
    global $artistpro_security;

    // Initialize all classes (create instances and call init())
    $artistpro_security['ip_manager'] = new ArtistPro_IP_Manager();
    $artistpro_security['ip_manager']->init();

    $artistpro_security['login_logger'] = new ArtistPro_Login_Logger();
    $artistpro_security['login_logger']->init();

    $artistpro_security['security_core'] = new ArtistPro_Security_Core();
    $artistpro_security['security_core']->init();

    $artistpro_security['geo_blocker'] = new ArtistPro_Geo_Blocker();
    $artistpro_security['geo_blocker']->init();

    $artistpro_security['blocklist_sync'] = new ArtistPro_Blocklist_Sync();
    $artistpro_security['blocklist_sync']->init();

    $artistpro_security['security_cron'] = new ArtistPro_Security_Cron();
    $artistpro_security['security_cron']->init();

    $artistpro_security['security_ajax'] = new ArtistPro_Security_Ajax();
    $artistpro_security['security_ajax']->init();

    $artistpro_security['security_notifier'] = new ArtistPro_Security_Notifier();
    $artistpro_security['security_notifier']->init();

    $artistpro_security['portal_integration'] = new ArtistPro_Portal_Integration();
    $artistpro_security['portal_integration']->init();
}
add_action('plugins_loaded', 'artistpro_security_init', 20);

/**
 * Register admin menu
 */
function artistpro_security_admin_menu() {
    add_options_page(
        'Security',
        'Security',
        'manage_options',
        'artistpro-security',
        'artistpro_security_dashboard_page'
    );
}
add_action('admin_menu', 'artistpro_security_admin_menu');

/**
 * Dashboard page callback
 */
function artistpro_security_dashboard_page() {
    $tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'dashboard';

    $tabs = array(
        'dashboard' => 'Dashboard',
        'logs' => 'Login Logs',
        'ip-lists' => 'IP Lists',
        'countries' => 'Country Blocking',
        'cloudflare' => 'Cloudflare',
        'settings' => 'Settings',
        'license' => 'License',
    );

    ?>
    <div class="wrap">
        <h1>artistpro Security</h1>

        <nav class="nav-tab-wrapper">
            <?php foreach ($tabs as $tab_id => $tab_name): ?>
                <a href="?page=artistpro-security&tab=<?php echo esc_attr($tab_id); ?>"
                   class="nav-tab <?php echo $tab === $tab_id ? 'nav-tab-active' : ''; ?>">
                    <?php echo esc_html($tab_name); ?>
                </a>
            <?php endforeach; ?>
        </nav>

        <div class="tab-content" style="margin-top: 20px;">
            <?php
            $page_file = ARTISTPRO_SECURITY_DIR . 'admin/pages/' . $tab . '.php';
            if (file_exists($page_file)) {
                include $page_file;
            } else {
                echo '<p>Page not found.</p>';
            }
            ?>
        </div>
    </div>
    <?php
}

/**
 * Enqueue admin assets
 */
function artistpro_security_admin_assets($hook) {
    if (strpos($hook, 'artistpro-security') === false) {
        return;
    }

    wp_enqueue_style(
        'artistpro-security-admin',
        ARTISTPRO_SECURITY_URL . 'assets/css/admin.css',
        array(),
        ARTISTPRO_SECURITY_VERSION
    );

    wp_enqueue_script(
        'artistpro-security-admin',
        ARTISTPRO_SECURITY_URL . 'assets/js/admin.js',
        array('jquery'),
        ARTISTPRO_SECURITY_VERSION,
        true
    );

    wp_localize_script('artistpro-security-admin', 'artistproSecurity', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('artistpro_security_admin'),
    ));
}
add_action('admin_enqueue_scripts', 'artistpro_security_admin_assets');
