<?php
/**
 * Security Settings
 *
 * @package ArtistPro_Security
 */

if (!defined('WPINC')) {
    die;
}

// Get current settings
$settings = [
    'allowed_retries' => get_option('artistpro_security_allowed_retries', 4),
    'short_lockout_duration' => get_option('artistpro_security_short_lockout_duration', 20),
    'long_lockout_duration' => get_option('artistpro_security_long_lockout_duration', 1440),
    'lockouts_before_long' => get_option('artistpro_security_lockouts_before_long', 4),
    'auto_blacklist_threshold' => get_option('artistpro_security_auto_blacklist_threshold', 10),
    'log_retention' => get_option('artistpro_security_log_retention', 30),
    'notify_lockouts' => get_option('artistpro_security_notify_lockouts', true),
    'notify_blocks' => get_option('artistpro_security_notify_blocks', true),
    'daily_summary' => get_option('artistpro_security_daily_summary', false),
    'notify_email' => get_option('artistpro_security_notify_email', get_option('admin_email')),
    'maxmind_key' => get_option('artistpro_security_maxmind_key', '')
];
?>

<div class="wrap artistpro-security-settings">
    <h1>Security Settings</h1>

    <form id="security-settings-form">
        <!-- Rate Limiting -->
        <div class="settings-section">
            <h2>Rate Limiting</h2>
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="allowed_retries">Allowed Retries</label>
                    </th>
                    <td>
                        <input type="number" id="allowed_retries" name="allowed_retries"
                               value="<?php echo esc_attr($settings['allowed_retries']); ?>"
                               min="1" max="20" class="small-text">
                        <p class="description">Number of failed login attempts before lockout</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="short_lockout_duration">Short Lockout Duration</label>
                    </th>
                    <td>
                        <input type="number" id="short_lockout_duration" name="short_lockout_duration"
                               value="<?php echo esc_attr($settings['short_lockout_duration']); ?>"
                               min="1" max="1440" class="small-text"> minutes
                        <p class="description">Duration of first lockout (default: 20 minutes)</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="long_lockout_duration">Long Lockout Duration</label>
                    </th>
                    <td>
                        <input type="number" id="long_lockout_duration" name="long_lockout_duration"
                               value="<?php echo esc_attr($settings['long_lockout_duration']); ?>"
                               min="60" max="10080" class="small-text"> minutes
                        <p class="description">Duration after repeated lockouts (default: 1440 = 24 hours)</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="lockouts_before_long">Lockouts Before Long</label>
                    </th>
                    <td>
                        <input type="number" id="lockouts_before_long" name="lockouts_before_long"
                               value="<?php echo esc_attr($settings['lockouts_before_long']); ?>"
                               min="1" max="10" class="small-text">
                        <p class="description">Short lockouts before escalating to long lockout</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="auto_blacklist_threshold">Auto-Blacklist Threshold</label>
                    </th>
                    <td>
                        <input type="number" id="auto_blacklist_threshold" name="auto_blacklist_threshold"
                               value="<?php echo esc_attr($settings['auto_blacklist_threshold']); ?>"
                               min="0" max="100" class="small-text">
                        <p class="description">Auto-blacklist after this many failed attempts (0 = disabled)</p>
                    </td>
                </tr>
            </table>
        </div>

        <!-- Logging -->
        <div class="settings-section">
            <h2>Logging</h2>
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="log_retention">Log Retention</label>
                    </th>
                    <td>
                        <input type="number" id="log_retention" name="log_retention"
                               value="<?php echo esc_attr($settings['log_retention']); ?>"
                               min="1" max="365" class="small-text"> days
                        <p class="description">How long to keep login attempt logs</p>
                    </td>
                </tr>
            </table>
        </div>

        <!-- Notifications -->
        <div class="settings-section">
            <h2>Email Notifications</h2>
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="notify_email">Notification Email</label>
                    </th>
                    <td>
                        <input type="email" id="notify_email" name="notify_email"
                               value="<?php echo esc_attr($settings['notify_email']); ?>"
                               class="regular-text">
                        <p class="description">Email address for security notifications</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">Notification Types</th>
                    <td>
                        <fieldset>
                            <label>
                                <input type="checkbox" name="notify_lockouts" value="1"
                                       <?php checked($settings['notify_lockouts']); ?>>
                                Send email on lockouts
                            </label><br>
                            <label>
                                <input type="checkbox" name="notify_blocks" value="1"
                                       <?php checked($settings['notify_blocks']); ?>>
                                Send email on permanent blocks
                            </label><br>
                            <label>
                                <input type="checkbox" name="daily_summary" value="1"
                                       <?php checked($settings['daily_summary']); ?>>
                                Send daily security summary
                            </label>
                        </fieldset>
                    </td>
                </tr>
            </table>
        </div>

        <!-- GeoIP -->
        <div class="settings-section">
            <h2>GeoIP (Country Blocking)</h2>
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="maxmind_key">MaxMind License Key</label>
                    </th>
                    <td>
                        <input type="text" id="maxmind_key" name="maxmind_key"
                               value="<?php echo esc_attr($settings['maxmind_key']); ?>"
                               class="regular-text" autocomplete="off">
                        <p class="description">
                            Required for country blocking.
                            <a href="https://dev.maxmind.com/geoip/geolite2-free-geolocation-data" target="_blank">Get a free GeoLite2 license key</a>
                        </p>
                        <?php if ($settings['maxmind_key']): ?>
                        <p>
                            <button type="button" class="button download-geoip">Download/Update GeoIP Database</button>
                            <?php
                            $last_update = get_option('artistpro_security_geoip_updated', 0);
                            if ($last_update) {
                                echo '<span class="geoip-status">Last updated: ' . human_time_diff($last_update) . ' ago</span>';
                            }
                            ?>
                        </p>
                        <?php endif; ?>
                    </td>
                </tr>
            </table>
        </div>

        <!-- Trusted Proxies -->
        <div class="settings-section">
            <h2>Advanced</h2>
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label>Trusted Proxy Headers</label>
                    </th>
                    <td>
                        <fieldset>
                            <label>
                                <input type="checkbox" name="trust_cf" value="1"
                                       <?php checked(get_option('artistpro_security_trust_cf', false)); ?>>
                                Trust Cloudflare (CF-Connecting-IP)
                            </label><br>
                            <label>
                                <input type="checkbox" name="trust_xff" value="1"
                                       <?php checked(get_option('artistpro_security_trust_xff', false)); ?>>
                                Trust X-Forwarded-For header
                            </label>
                        </fieldset>
                        <p class="description">Enable if your site is behind a proxy or CDN</p>
                    </td>
                </tr>
            </table>
        </div>

        <p class="submit">
            <button type="submit" class="button button-primary">Save Settings</button>
            <span class="spinner"></span>
            <span class="save-status"></span>
        </p>
    </form>

    <!-- Debug Info -->
    <div class="settings-section debug-info">
        <h2>Debug Information</h2>
        <table class="form-table">
            <tr>
                <th>Your IP Address</th>
                <td><code><?php echo esc_html((new ArtistPro_IP_Manager())->get_client_ip()); ?></code></td>
            </tr>
            <tr>
                <th>Plugin Version</th>
                <td><?php echo ARTISTPRO_SECURITY_VERSION; ?></td>
            </tr>
            <tr>
                <th>GeoIP Database</th>
                <td>
                    <?php
                    $geoip_path = ARTISTPRO_SECURITY_PATH . 'data/GeoLite2-Country.mmdb';
                    if (file_exists($geoip_path)) {
                        echo '<span style="color:green;">✓ Installed</span> (' . size_format(filesize($geoip_path)) . ')';
                    } else {
                        echo '<span style="color:red;">✗ Not installed</span>';
                    }
                    ?>
                </td>
            </tr>
            <tr>
                <th>Next Blocklist Sync</th>
                <td>
                    <?php
                    $next = wp_next_scheduled('artistpro_security_sync_blocklists');
                    echo $next ? human_time_diff($next) . ' from now' : 'Not scheduled';
                    ?>
                </td>
            </tr>
        </table>
    </div>
</div>
