<?php
/**
 * Security Logs - Login attempt viewer
 *
 * @package ArtistPro_Security
 */

if (!defined('WPINC')) {
    die;
}

global $wpdb;
$table = $wpdb->prefix . 'artistpro_security_logs';

// Pagination
$per_page = 50;
$page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
$offset = ($page - 1) * $per_page;

// Filters
$filter_result = isset($_GET['result']) ? sanitize_text_field($_GET['result']) : '';
$filter_ip = isset($_GET['ip']) ? sanitize_text_field($_GET['ip']) : '';
$filter_type = isset($_GET['type']) ? sanitize_text_field($_GET['type']) : '';

// Build query
$where = ['1=1'];
$values = [];

if ($filter_result) {
    $where[] = 'result = %s';
    $values[] = $filter_result;
}

if ($filter_ip) {
    $where[] = 'ip_address LIKE %s';
    $values[] = '%' . $wpdb->esc_like($filter_ip) . '%';
}

if ($filter_type) {
    $where[] = 'login_type = %s';
    $values[] = $filter_type;
}

$where_sql = implode(' AND ', $where);

// Get total count
$count_sql = "SELECT COUNT(*) FROM {$table} WHERE {$where_sql}";
if (!empty($values)) {
    $total = $wpdb->get_var($wpdb->prepare($count_sql, $values));
} else {
    $total = $wpdb->get_var($count_sql);
}
$total_pages = ceil($total / $per_page);

// Get logs
$query = "SELECT * FROM {$table} WHERE {$where_sql} ORDER BY created_at DESC LIMIT %d OFFSET %d";
$query_values = array_merge($values, [$per_page, $offset]);
$logs = $wpdb->get_results($wpdb->prepare($query, $query_values));

// Get unique values for filters
$result_types = $wpdb->get_col("SELECT DISTINCT result FROM {$table}");
$login_types = $wpdb->get_col("SELECT DISTINCT login_type FROM {$table}");
?>

<div class="wrap artistpro-security-logs">
    <h1>Security Logs</h1>

    <!-- Filters -->
    <div class="tablenav top">
        <form method="get" class="filters-form">
            <input type="hidden" name="page" value="artistpro-security">
            <input type="hidden" name="tab" value="logs">

            <select name="result">
                <option value="">All Results</option>
                <?php foreach ($result_types as $type): ?>
                <option value="<?php echo esc_attr($type); ?>" <?php selected($filter_result, $type); ?>>
                    <?php echo esc_html(ucfirst($type)); ?>
                </option>
                <?php endforeach; ?>
            </select>

            <select name="type">
                <option value="">All Login Types</option>
                <?php foreach ($login_types as $type): ?>
                <option value="<?php echo esc_attr($type); ?>" <?php selected($filter_type, $type); ?>>
                    <?php echo esc_html($type); ?>
                </option>
                <?php endforeach; ?>
            </select>

            <input type="text" name="ip" placeholder="Filter by IP" value="<?php echo esc_attr($filter_ip); ?>">

            <button type="submit" class="button">Filter</button>
            <a href="<?php echo admin_url('options-general.php?page=artistpro-security&tab=logs'); ?>" class="button">Clear</a>
        </form>

        <div class="tablenav-pages">
            <span class="displaying-num"><?php echo number_format($total); ?> items</span>
        </div>

        <div class="log-actions">
            <button class="button export-logs">Export CSV</button>
            <button class="button button-link-delete clear-logs">Clear Old Logs</button>
        </div>
    </div>

    <!-- Logs Table -->
    <table class="wp-list-table widefat fixed striped">
        <thead>
            <tr>
                <th style="width:150px;">Time</th>
                <th style="width:130px;">IP Address</th>
                <th style="width:50px;">Country</th>
                <th>Username</th>
                <th style="width:120px;">Type</th>
                <th style="width:80px;">Result</th>
                <th>Reason</th>
                <th style="width:100px;">Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($logs)): ?>
            <tr>
                <td colspan="8" style="text-align:center;padding:20px;">No log entries found.</td>
            </tr>
            <?php else: ?>
                <?php foreach ($logs as $log): ?>
                <tr>
                    <td>
                        <span title="<?php echo esc_attr($log->created_at); ?>">
                            <?php echo human_time_diff(strtotime($log->created_at)) . ' ago'; ?>
                        </span>
                    </td>
                    <td><code><?php echo esc_html($log->ip_address); ?></code></td>
                    <td><?php echo esc_html($log->country_code ?: '-'); ?></td>
                    <td><?php echo esc_html($log->username ?: '-'); ?></td>
                    <td><span class="badge badge-secondary"><?php echo esc_html($log->login_type); ?></span></td>
                    <td>
                        <span class="badge badge-<?php
                            echo $log->result === 'success' ? 'success' :
                                ($log->result === 'blocked' ? 'danger' :
                                ($log->result === 'lockout' ? 'danger' : 'warning'));
                        ?>">
                            <?php echo esc_html($log->result); ?>
                        </span>
                    </td>
                    <td class="failure-reason"><?php echo esc_html($log->failure_reason ?: '-'); ?></td>
                    <td>
                        <button class="button button-small block-ip" data-ip="<?php echo esc_attr($log->ip_address); ?>" title="Block IP">
                            <span class="dashicons dashicons-shield"></span>
                        </button>
                        <button class="button button-small whitelist-ip" data-ip="<?php echo esc_attr($log->ip_address); ?>" title="Whitelist IP">
                            <span class="dashicons dashicons-yes"></span>
                        </button>
                    </td>
                </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>

    <!-- Pagination -->
    <?php if ($total_pages > 1): ?>
    <div class="tablenav bottom">
        <div class="tablenav-pages">
            <?php
            $page_links = paginate_links([
                'base' => add_query_arg('paged', '%#%'),
                'format' => '',
                'prev_text' => '&laquo;',
                'next_text' => '&raquo;',
                'total' => $total_pages,
                'current' => $page
            ]);
            echo $page_links;
            ?>
        </div>
    </div>
    <?php endif; ?>
</div>

<!-- Clear Logs Modal -->
<div id="clear-logs-modal" class="modal" style="display:none;">
    <div class="modal-content">
        <h3>Clear Security Logs</h3>
        <p>Choose how many days of logs to keep:</p>
        <select id="clear-logs-days">
            <option value="30">Keep last 30 days</option>
            <option value="7">Keep last 7 days</option>
            <option value="1">Keep last 24 hours</option>
            <option value="0">Clear all logs</option>
        </select>
        <div class="modal-actions">
            <button class="button button-primary confirm-clear-logs">Clear Logs</button>
            <button class="button cancel-modal">Cancel</button>
        </div>
    </div>
</div>
