<?php
/**
 * License Settings Page
 * 
 * @package artistproSecurity
 */

if (!defined('ABSPATH')) {
    exit;
}

// Get license info
$license_key = get_option('artistpro_security_license_key', '');
$license_status = get_option('artistpro_security_license_status', 'inactive');
$license_expires = get_option('artistpro_security_license_expires', '');
$license_activations = get_option('artistpro_security_license_activations', 0);
$license_limit = get_option('artistpro_security_license_limit', 0);

// Handle form submission
if (isset($_POST['artistpro_save_license']) && check_admin_referer('artistpro_license_settings', 'artistpro_license_nonce')) {
    update_option('artistpro_security_license_key', sanitize_text_field($_POST['license_key']));
    echo '<div class="notice notice-success"><p>License key saved.</p></div>';
}
?>

<div class="wrap artistpro-security-wrap">
    <h1>License Settings</h1>
    
    <div class="artistpro-card">
        <h2>License Activation</h2>
        
        <?php if ($license_status === 'active'): ?>
            <div class="notice notice-success inline" style="margin: 15px 0;">
                <p><strong>✅ License Active</strong></p>
                <?php if ($license_expires): ?>
                    <p>Expires: <?php echo date('F j, Y', strtotime($license_expires)); ?></p>
                <?php else: ?>
                    <p>Lifetime License</p>
                <?php endif; ?>
                <?php if ($license_limit > 0): ?>
                    <p>Activations: <?php echo $license_activations; ?> / <?php echo $license_limit; ?></p>
                <?php endif; ?>
            </div>
        <?php else: ?>
            <div class="notice notice-warning inline" style="margin: 15px 0;">
                <p><strong>⚠️ License Inactive</strong></p>
                <p>Enter your license key below to activate updates and support.</p>
            </div>
        <?php endif; ?>
        
        <form method="post" action="">
            <?php wp_nonce_field('artistpro_license_settings', 'artistpro_license_nonce'); ?>
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="license_key">License Key</label>
                    </th>
                    <td>
                        <input type="text" 
                               id="license_key" 
                               name="license_key" 
                               value="<?php echo esc_attr($license_key); ?>" 
                               class="regular-text code"
                               placeholder="XXXX-XXXX-XXXX-XXXX-XXXX-XXXX">
                        
                        <p class="description">
                            Enter the license key you received after purchase.
                            <br>Don't have a license? 
                            <a href="https://artistprowp.com/products/artistpro-security/" target="_blank">Purchase one here</a>
                        </p>
                    </td>
                </tr>
            </table>
            
            <p class="submit">
                <button type="submit" name="artistpro_save_license" class="button button-primary">
                    Save License Key
                </button>
                
                <?php if ($license_key): ?>
                    <button type="button" id="activate-license" class="button button-secondary">
                        <?php echo $license_status === 'active' ? 'Revalidate License' : 'Activate License'; ?>
                    </button>
                    
                    <?php if ($license_status === 'active'): ?>
                        <button type="button" id="deactivate-license" class="button">
                            Deactivate License
                        </button>
                    <?php endif; ?>
                    
                    <button type="button" id="check-updates" class="button">
                        Check for Updates
                    </button>
                <?php endif; ?>
            </p>
        </form>
        
        <div id="license-response" style="margin-top: 20px;"></div>
    </div>
    
    <div class="artistpro-card">
        <h2>Product Information</h2>
        <table class="wp-list-table widefat fixed striped">
            <tr>
                <th style="width: 200px;">Product Name</th>
                <td><strong>artistpro Security</strong></td>
            </tr>
            <tr>
                <th>Current Version</th>
                <td>1.1.0</td>
            </tr>
            <tr>
                <th>Product ID</th>
                <td><code>artistprosecurity</code></td>
            </tr>
            <tr>
                <th>Update Server</th>
                <td><code>https://artistprowp.com/wp-json/artistpro-license/v1/</code></td>
            </tr>
        </table>
    </div>
</div>

<script>
jQuery(document).ready(function($) {
    function showMessage(message, type) {
        var className = type === 'success' ? 'notice-success' : 'notice-error';
        $('#license-response').html(
            '<div class="notice ' + className + ' inline"><p>' + message + '</p></div>'
        );
    }
    
    $('#activate-license').on('click', function() {
        var button = $(this);
        var originalText = button.text();
        button.prop('disabled', true).text('Activating...');
        
        $.post('https://artistprowp.com/wp-json/artistpro-license/v1/activate', {
            license_key: $('#license_key').val(),
            product_id: 'artistprosecurity',
            site_url: '<?php echo home_url(); ?>'
        }, function(response) {
            if (response.success) {
                showMessage('✅ ' + response.message, 'success');
                location.reload();
            } else {
                showMessage('❌ ' + response.message, 'error');
            }
        }).fail(function() {
            showMessage('❌ Connection failed. Please try again.', 'error');
        }).always(function() {
            button.prop('disabled', false).text(originalText);
        });
    });
    
    $('#deactivate-license').on('click', function() {
        if (!confirm('Are you sure you want to deactivate this license on this site?')) {
            return;
        }
        
        var button = $(this);
        var originalText = button.text();
        button.prop('disabled', true).text('Deactivating...');
        
        $.post('https://artistprowp.com/wp-json/artistpro-license/v1/deactivate', {
            license_key: $('#license_key').val(),
            site_url: '<?php echo home_url(); ?>'
        }, function(response) {
            if (response.success) {
                showMessage('✅ ' + response.message, 'success');
                location.reload();
            } else {
                showMessage('❌ ' + response.message, 'error');
            }
        }).fail(function() {
            showMessage('❌ Connection failed. Please try again.', 'error');
        }).always(function() {
            button.prop('disabled', false).text(originalText);
        });
    });
    
    $('#check-updates').on('click', function() {
        var button = $(this);
        var originalText = button.text();
        button.prop('disabled', true).text('Checking...');
        
        $.post(ajaxurl, {
            action: 'artistpro_check_updates',
            nonce: '<?php echo wp_create_nonce('artistpro_check_updates'); ?>'
        }, function(response) {
            if (response.success) {
                if (response.data.update_available) {
                    showMessage('🎉 Update available: v' + response.data.new_version, 'success');
                } else {
                    showMessage('✅ You are running the latest version!', 'success');
                }
            } else {
                showMessage('❌ Could not check for updates.', 'error');
            }
        }).always(function() {
            button.prop('disabled', false).text(originalText);
        });
    });
});
</script>
