<?php
/**
 * IP Lists - Whitelist/Blacklist management
 *
 * @package ArtistPro_Security
 */

if (!defined('WPINC')) {
    die;
}

global $wpdb;
$table = $wpdb->prefix . 'artistpro_security_ip_lists';
$lockouts_table = $wpdb->prefix . 'artistpro_security_lockouts';

// Get current client IP
$ip_manager = new ArtistPro_IP_Manager();
$client_ip = $ip_manager->get_client_ip();

// Get lists
$whitelist = $wpdb->get_results("SELECT * FROM {$table} WHERE list_type = 'whitelist' ORDER BY source, ip_address");
$blacklist_manual = $wpdb->get_results("SELECT * FROM {$table} WHERE list_type = 'blacklist' AND source IN ('manual', 'auto_blocked') ORDER BY ip_address");
$active_lockouts = $wpdb->get_results("SELECT * FROM {$lockouts_table} WHERE locked_until > NOW() OR lockout_type = 'permanent' ORDER BY created_at DESC");

// Blocklist counts
$blocklist_sync = new ArtistPro_Blocklist_Sync();
$blocklist_status = $blocklist_sync->get_sync_status();
?>

<div class="wrap artistpro-security-ip-lists">
    <h1>IP Management</h1>

    <div class="ip-lists-grid">
        <!-- Add IP Form -->
        <div class="ip-card add-ip-card">
            <h2>Add IP Address</h2>
            <form id="add-ip-form">
                <p>
                    <label>IP Address or CIDR</label>
                    <input type="text" name="ip" placeholder="192.168.1.1 or 10.0.0.0/8" required>
                </p>
                <p>
                    <label>List Type</label>
                    <select name="list_type">
                        <option value="whitelist">Whitelist (Always Allow)</option>
                        <option value="blacklist">Blacklist (Always Block)</option>
                    </select>
                </p>
                <p>
                    <label>Reason (optional)</label>
                    <input type="text" name="reason" placeholder="e.g., Office IP, Known attacker">
                </p>
                <p>
                    <label>Expires (optional)</label>
                    <input type="datetime-local" name="expires">
                </p>
                <p>
                    <button type="submit" class="button button-primary">Add IP</button>
                </p>
            </form>
            <div class="current-ip">
                <small>Your current IP: <code><?php echo esc_html($client_ip); ?></code></small>
                <button class="button button-small whitelist-self" data-ip="<?php echo esc_attr($client_ip); ?>">Whitelist My IP</button>
            </div>
        </div>

        <!-- Whitelist -->
        <div class="ip-card">
            <h2>Whitelist <span class="count">(<?php echo count($whitelist); ?>)</span></h2>
            <p class="description">IPs that will never be blocked</p>

            <?php if (empty($whitelist)): ?>
            <p class="no-data">No whitelisted IPs</p>
            <?php else: ?>
            <table class="ip-table">
                <thead>
                    <tr>
                        <th>IP Address</th>
                        <th>Reason</th>
                        <th>Expires</th>
                        <th></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($whitelist as $ip): ?>
                    <tr>
                        <td><code><?php echo esc_html($ip->ip_address); ?></code></td>
                        <td><?php echo esc_html($ip->reason ?: '-'); ?></td>
                        <td><?php echo $ip->expires_at ? date('M j, Y', strtotime($ip->expires_at)) : 'Never'; ?></td>
                        <td>
                            <button class="button button-small remove-ip" data-ip="<?php echo esc_attr($ip->ip_address); ?>">Remove</button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>

        <!-- Manual Blacklist -->
        <div class="ip-card">
            <h2>Blacklist <span class="count">(<?php echo count($blacklist_manual); ?>)</span></h2>
            <p class="description">Manually blocked IPs</p>

            <?php if (empty($blacklist_manual)): ?>
            <p class="no-data">No manually blacklisted IPs</p>
            <?php else: ?>
            <table class="ip-table">
                <thead>
                    <tr>
                        <th>IP Address</th>
                        <th>Source</th>
                        <th>Reason</th>
                        <th></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($blacklist_manual as $ip): ?>
                    <tr>
                        <td><code><?php echo esc_html($ip->ip_address); ?></code></td>
                        <td><span class="badge badge-<?php echo $ip->source === 'manual' ? 'secondary' : 'warning'; ?>"><?php echo esc_html($ip->source); ?></span></td>
                        <td><?php echo esc_html($ip->reason ?: '-'); ?></td>
                        <td>
                            <button class="button button-small remove-ip" data-ip="<?php echo esc_attr($ip->ip_address); ?>">Remove</button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>

        <!-- Active Lockouts -->
        <div class="ip-card">
            <h2>Active Lockouts <span class="count">(<?php echo count($active_lockouts); ?>)</span></h2>
            <p class="description">Currently locked out IPs</p>

            <?php if (empty($active_lockouts)): ?>
            <p class="no-data">No active lockouts</p>
            <?php else: ?>
            <table class="ip-table">
                <thead>
                    <tr>
                        <th>IP Address</th>
                        <th>Type</th>
                        <th>Attempts</th>
                        <th>Expires</th>
                        <th></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($active_lockouts as $lockout): ?>
                    <tr>
                        <td><code><?php echo esc_html($lockout->ip_address); ?></code></td>
                        <td>
                            <span class="badge badge-<?php echo $lockout->lockout_type === 'permanent' ? 'danger' : ($lockout->lockout_type === 'long' ? 'warning' : 'secondary'); ?>">
                                <?php echo esc_html($lockout->lockout_type); ?>
                            </span>
                        </td>
                        <td><?php echo esc_html($lockout->attempts_count); ?></td>
                        <td>
                            <?php
                            if ($lockout->lockout_type === 'permanent') {
                                echo 'Never';
                            } else {
                                $remaining = strtotime($lockout->locked_until) - time();
                                if ($remaining > 3600) {
                                    echo round($remaining / 3600, 1) . ' hours';
                                } else {
                                    echo round($remaining / 60) . ' min';
                                }
                            }
                            ?>
                        </td>
                        <td>
                            <button class="button button-small unlock-ip" data-ip="<?php echo esc_attr($lockout->ip_address); ?>">Unlock</button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php endif; ?>
        </div>

        <!-- External Blocklists -->
        <div class="ip-card full-width">
            <h2>External Blocklists</h2>
            <p class="description">IP ranges from threat intelligence sources</p>

            <table class="ip-table blocklist-table">
                <thead>
                    <tr>
                        <th>Source</th>
                        <th>Description</th>
                        <th>IPs Loaded</th>
                        <th>Last Sync</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($blocklist_status as $key => $source): ?>
                    <tr>
                        <td><strong><?php echo esc_html($source['name']); ?></strong></td>
                        <td><?php echo esc_html($source['description']); ?></td>
                        <td><?php echo number_format($source['ip_count']); ?></td>
                        <td><?php echo esc_html($source['last_sync_human']); ?></td>
                        <td>
                            <label class="switch">
                                <input type="checkbox" class="toggle-blocklist" data-source="<?php echo esc_attr($key); ?>" <?php checked($source['enabled']); ?>>
                                <span class="slider"></span>
                            </label>
                        </td>
                        <td>
                            <button class="button button-small sync-blocklist" data-source="<?php echo esc_attr($key); ?>" <?php disabled(!$source['enabled']); ?>>
                                Sync Now
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>
