<?php
/**
 * Security Dashboard - Overview and stats
 *
 * @package ArtistPro_Security
 */

if (!defined('WPINC')) {
    die;
}

global $wpdb;
$logs_table = $wpdb->prefix . 'artistpro_security_logs';
$lockouts_table = $wpdb->prefix . 'artistpro_security_lockouts';
$ip_lists_table = $wpdb->prefix . 'artistpro_security_ip_lists';

// Get stats
$today_stats = $wpdb->get_row("
    SELECT
        COUNT(*) as total,
        SUM(CASE WHEN result = 'success' THEN 1 ELSE 0 END) as successful,
        SUM(CASE WHEN result = 'failed' THEN 1 ELSE 0 END) as failed,
        SUM(CASE WHEN result = 'blocked' THEN 1 ELSE 0 END) as blocked,
        SUM(CASE WHEN result = 'lockout' THEN 1 ELSE 0 END) as lockouts
    FROM {$logs_table}
    WHERE created_at > DATE_SUB(NOW(), INTERVAL 24 HOUR)
");

$week_stats = $wpdb->get_row("
    SELECT
        COUNT(*) as total,
        SUM(CASE WHEN result = 'success' THEN 1 ELSE 0 END) as successful,
        SUM(CASE WHEN result = 'failed' THEN 1 ELSE 0 END) as failed,
        SUM(CASE WHEN result = 'blocked' THEN 1 ELSE 0 END) as blocked
    FROM {$logs_table}
    WHERE created_at > DATE_SUB(NOW(), INTERVAL 7 DAY)
");

$active_lockouts = $wpdb->get_var("SELECT COUNT(*) FROM {$lockouts_table} WHERE locked_until > NOW() OR lockout_type = 'permanent'");
$blacklisted_ips = $wpdb->get_var("SELECT COUNT(*) FROM {$ip_lists_table} WHERE list_type = 'blacklist'");
$whitelisted_ips = $wpdb->get_var("SELECT COUNT(*) FROM {$ip_lists_table} WHERE list_type = 'whitelist' AND source = 'manual'");

// Recent activity
$recent_logs = $wpdb->get_results("
    SELECT * FROM {$logs_table}
    ORDER BY created_at DESC
    LIMIT 10
");

// Top attackers
$top_attackers = $wpdb->get_results("
    SELECT ip_address, COUNT(*) as attempts, MAX(created_at) as last_attempt
    FROM {$logs_table}
    WHERE result IN ('failed', 'blocked', 'lockout')
      AND created_at > DATE_SUB(NOW(), INTERVAL 24 HOUR)
    GROUP BY ip_address
    ORDER BY attempts DESC
    LIMIT 5
");

// Blocklist status
$blocklist_sync = new ArtistPro_Blocklist_Sync();
$blocklist_status = $blocklist_sync->get_sync_status();
$blocklist_total = $blocklist_sync->get_total_count();
?>

<div class="wrap artistpro-security-dashboard">
    <h1>Security Dashboard</h1>

    <!-- Stats Cards -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon success"><span class="dashicons dashicons-yes-alt"></span></div>
            <div class="stat-content">
                <div class="stat-number"><?php echo number_format($today_stats->successful ?? 0); ?></div>
                <div class="stat-label">Successful Logins (24h)</div>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon warning"><span class="dashicons dashicons-warning"></span></div>
            <div class="stat-content">
                <div class="stat-number"><?php echo number_format($today_stats->failed ?? 0); ?></div>
                <div class="stat-label">Failed Attempts (24h)</div>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon danger"><span class="dashicons dashicons-shield"></span></div>
            <div class="stat-content">
                <div class="stat-number"><?php echo number_format($today_stats->blocked ?? 0); ?></div>
                <div class="stat-label">Blocked (24h)</div>
            </div>
        </div>
        <div class="stat-card">
            <div class="stat-icon info"><span class="dashicons dashicons-lock"></span></div>
            <div class="stat-content">
                <div class="stat-number"><?php echo number_format($active_lockouts); ?></div>
                <div class="stat-label">Active Lockouts</div>
            </div>
        </div>
    </div>

    <div class="dashboard-grid">
        <!-- Recent Activity -->
        <div class="dashboard-card">
            <div class="card-header">
                <h2>Recent Activity</h2>
                <a href="<?php echo admin_url('options-general.php?page=artistpro-security&tab=logs'); ?>" class="button button-small">View All</a>
            </div>
            <div class="card-content">
                <?php if (empty($recent_logs)): ?>
                    <p class="no-data">No login activity recorded yet.</p>
                <?php else: ?>
                    <table class="activity-table">
                        <thead>
                            <tr>
                                <th>Time</th>
                                <th>IP</th>
                                <th>User</th>
                                <th>Type</th>
                                <th>Result</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_logs as $log): ?>
                            <tr>
                                <td><?php echo human_time_diff(strtotime($log->created_at)) . ' ago'; ?></td>
                                <td><code><?php echo esc_html($log->ip_address); ?></code></td>
                                <td><?php echo esc_html($log->username ?: '-'); ?></td>
                                <td><?php echo esc_html($log->login_type); ?></td>
                                <td>
                                    <span class="badge badge-<?php echo $log->result === 'success' ? 'success' : ($log->result === 'blocked' ? 'danger' : 'warning'); ?>">
                                        <?php echo esc_html($log->result); ?>
                                    </span>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>

        <!-- Top Attackers -->
        <div class="dashboard-card">
            <div class="card-header">
                <h2>Suspicious IPs (24h)</h2>
            </div>
            <div class="card-content">
                <?php if (empty($top_attackers)): ?>
                    <p class="no-data">No suspicious activity detected.</p>
                <?php else: ?>
                    <table class="activity-table">
                        <thead>
                            <tr>
                                <th>IP Address</th>
                                <th>Attempts</th>
                                <th>Last Seen</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($top_attackers as $attacker): ?>
                            <tr>
                                <td><code><?php echo esc_html($attacker->ip_address); ?></code></td>
                                <td><strong><?php echo number_format($attacker->attempts); ?></strong></td>
                                <td><?php echo human_time_diff(strtotime($attacker->last_attempt)) . ' ago'; ?></td>
                                <td>
                                    <button class="button button-small block-ip" data-ip="<?php echo esc_attr($attacker->ip_address); ?>">
                                        Block
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>

        <!-- Protection Status -->
        <div class="dashboard-card">
            <div class="card-header">
                <h2>Protection Status</h2>
            </div>
            <div class="card-content">
                <ul class="status-list">
                    <li>
                        <span class="status-icon active"></span>
                        <span class="status-text">Login Rate Limiting</span>
                        <span class="status-detail">Active</span>
                    </li>
                    <li>
                        <span class="status-icon <?php echo $blocklist_total > 0 ? 'active' : 'inactive'; ?>"></span>
                        <span class="status-text">IP Blocklists</span>
                        <span class="status-detail"><?php echo number_format($blocklist_total); ?> IPs blocked</span>
                    </li>
                    <li>
                        <span class="status-icon <?php echo file_exists(ARTISTPRO_SECURITY_PATH . 'data/GeoLite2-Country.mmdb') ? 'active' : 'inactive'; ?>"></span>
                        <span class="status-text">Country Blocking</span>
                        <span class="status-detail"><?php echo file_exists(ARTISTPRO_SECURITY_PATH . 'data/GeoLite2-Country.mmdb') ? 'GeoIP Ready' : 'No GeoIP Database'; ?></span>
                    </li>
                    <li>
                        <span class="status-icon active"></span>
                        <span class="status-text">Login Logging</span>
                        <span class="status-detail"><?php echo number_format($week_stats->total ?? 0); ?> events (7 days)</span>
                    </li>
                </ul>
            </div>
        </div>

        <!-- Blocklist Status -->
        <div class="dashboard-card">
            <div class="card-header">
                <h2>Blocklist Sources</h2>
                <button class="button button-small sync-all-blocklists">Sync Now</button>
            </div>
            <div class="card-content">
                <table class="activity-table">
                    <thead>
                        <tr>
                            <th>Source</th>
                            <th>Status</th>
                            <th>IPs</th>
                            <th>Last Sync</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($blocklist_status as $key => $source): ?>
                        <tr>
                            <td><?php echo esc_html($source['name']); ?></td>
                            <td>
                                <span class="badge badge-<?php echo $source['enabled'] ? 'success' : 'secondary'; ?>">
                                    <?php echo $source['enabled'] ? 'Enabled' : 'Disabled'; ?>
                                </span>
                            </td>
                            <td><?php echo number_format($source['ip_count']); ?></td>
                            <td><?php echo esc_html($source['last_sync_human']); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Quick Stats Footer -->
    <div class="quick-stats">
        <div class="quick-stat">
            <strong><?php echo number_format($blacklisted_ips); ?></strong> Blacklisted IPs
        </div>
        <div class="quick-stat">
            <strong><?php echo number_format($whitelisted_ips); ?></strong> Whitelisted IPs
        </div>
        <div class="quick-stat">
            <strong><?php echo number_format($week_stats->total ?? 0); ?></strong> Login Attempts (7 days)
        </div>
    </div>
</div>
