<?php
/**
 * Country Blocking - GeoIP country blocking
 *
 * @package ArtistPro_Security
 */

if (!defined('WPINC')) {
    die;
}

$geo_blocker = new ArtistPro_Geo_Blocker();
$countries = $geo_blocker->get_all_countries();
$stats = $geo_blocker->get_stats();

// Group countries by first letter
$country_groups = [];
foreach ($countries as $code => $country) {
    $first_letter = strtoupper(substr($country['name'], 0, 1));
    $country_groups[$first_letter][$code] = $country;
}
ksort($country_groups);

// Get blocked countries count
$blocked_count = $stats['blocked_countries_count'];
$has_geoip = $stats['database_exists'];
?>

<div class="wrap artistpro-security-country-blocking">
    <h1>Country Blocking</h1>

    <?php if (!$has_geoip): ?>
    <div class="notice notice-warning">
        <p>
            <strong>GeoIP Database Not Found</strong><br>
            To enable country blocking, you need to configure a MaxMind license key.
            <a href="https://dev.maxmind.com/geoip/geolite2-free-geolocation-data" target="_blank">Get a free GeoLite2 license key</a>
        </p>
        <p>
            <a href="<?php echo admin_url('options-general.php?page=artistpro-security&tab=settings'); ?>" class="button">Configure MaxMind Key</a>
        </p>
    </div>
    <?php endif; ?>

    <div class="country-stats">
        <div class="stat">
            <span class="number"><?php echo $blocked_count; ?></span>
            <span class="label">Countries Blocked</span>
        </div>
        <div class="stat">
            <span class="number"><?php echo count($countries); ?></span>
            <span class="label">Total Countries</span>
        </div>
        <div class="stat">
            <span class="number <?php echo $has_geoip ? 'success' : 'warning'; ?>">
                <?php echo $has_geoip ? 'Active' : 'Inactive'; ?>
            </span>
            <span class="label">GeoIP Database</span>
        </div>
    </div>

    <?php if (!empty($stats['blocked_by_country'])): ?>
    <div class="blocked-stats-card">
        <h3>Top Blocked Countries (Last 30 Days)</h3>
        <div class="blocked-countries-chart">
            <?php foreach ($stats['blocked_by_country'] as $row): ?>
            <div class="blocked-country-bar">
                <span class="country-code"><?php echo esc_html($row['country_code']); ?></span>
                <div class="bar" style="width: <?php echo min(100, ($row['count'] / max(array_column($stats['blocked_by_country'], 'count'))) * 100); ?>%"></div>
                <span class="count"><?php echo number_format($row['count']); ?></span>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
    <?php endif; ?>

    <div class="country-blocking-grid">
        <!-- Quick Actions -->
        <div class="quick-actions-card">
            <h3>Quick Actions</h3>
            <div class="quick-actions">
                <button class="button block-high-risk" data-countries="CN,RU,KP,IR">
                    Block High-Risk Countries
                    <small>(China, Russia, N. Korea, Iran)</small>
                </button>
                <button class="button unblock-all">
                    Unblock All Countries
                </button>
            </div>
            <p class="description">
                <strong>Note:</strong> Country blocking should be used carefully. Legitimate users from blocked countries will be unable to access your site.
            </p>
        </div>

        <!-- Country List -->
        <div class="country-list-card">
            <h3>Select Countries to Block</h3>

            <div class="country-search">
                <input type="text" id="country-search" placeholder="Search countries...">
            </div>

            <div class="country-list">
                <?php foreach ($country_groups as $letter => $group): ?>
                <div class="country-group">
                    <h4><?php echo $letter; ?></h4>
                    <div class="countries">
                        <?php foreach ($group as $code => $country): ?>
                        <label class="country-item <?php echo $country['blocked'] ? 'blocked' : ''; ?>">
                            <input type="checkbox"
                                   class="country-checkbox"
                                   data-code="<?php echo esc_attr($code); ?>"
                                   <?php checked($country['blocked']); ?>>
                            <span class="country-flag"><?php echo $this->get_flag_emoji($code); ?></span>
                            <span class="country-name"><?php echo esc_html($country['name']); ?></span>
                            <span class="country-code">(<?php echo $code; ?>)</span>
                        </label>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
</div>

<script>
// Country search filter
document.getElementById('country-search').addEventListener('input', function(e) {
    const search = e.target.value.toLowerCase();
    document.querySelectorAll('.country-item').forEach(function(item) {
        const name = item.querySelector('.country-name').textContent.toLowerCase();
        const code = item.querySelector('.country-code').textContent.toLowerCase();
        item.style.display = (name.includes(search) || code.includes(search)) ? '' : 'none';
    });
});
</script>

<?php
// Helper function for flag emoji
function get_flag_emoji($country_code) {
    $country_code = strtoupper($country_code);
    $flag = '';
    for ($i = 0; $i < strlen($country_code); $i++) {
        $flag .= mb_convert_encoding('&#' . (127397 + ord($country_code[$i])) . ';', 'UTF-8', 'HTML-ENTITIES');
    }
    return $flag;
}
?>
