<?php
/**
 * Cloudflare Integration Settings Page
 *
 * @package ArtistPro_Security
 */

if (!defined('WPINC')) {
    die;
}

// Get current settings
$enabled = get_option('artistpro_security_cloudflare_enabled', false);
$api_token = get_option('artistpro_security_cloudflare_api_token', '');
$zone_id = get_option('artistpro_security_cloudflare_zone_id', '');
$threat_check = get_option('artistpro_security_cloudflare_threat_check', false);
$threat_threshold = get_option('artistpro_security_cloudflare_threat_threshold', 80);
$auto_sync = get_option('artistpro_security_cloudflare_auto_sync', true);
$under_attack = get_option('artistpro_security_cloudflare_under_attack', false);

// Handle form submission
if (isset($_POST['submit']) && check_admin_referer('artistpro_security_cloudflare')) {
    update_option('artistpro_security_cloudflare_enabled', isset($_POST['enabled']));
    update_option('artistpro_security_cloudflare_api_token', sanitize_text_field($_POST['api_token']));
    update_option('artistpro_security_cloudflare_zone_id', sanitize_text_field($_POST['zone_id']));
    update_option('artistpro_security_cloudflare_threat_check', isset($_POST['threat_check']));
    update_option('artistpro_security_cloudflare_threat_threshold', (int) $_POST['threat_threshold']);
    update_option('artistpro_security_cloudflare_auto_sync', isset($_POST['auto_sync']));
    
    echo '<div class="notice notice-success"><p>Cloudflare settings saved!</p></div>';
    
    // Refresh values
    $enabled = isset($_POST['enabled']);
    $api_token = sanitize_text_field($_POST['api_token']);
    $zone_id = sanitize_text_field($_POST['zone_id']);
}
?>

<div class="wrap artistpro-security-cloudflare">
    <h2>⚡ Cloudflare Integration</h2>
    <p class="description">
        <strong>PREMIUM FEATURE:</strong> Block attacks at Cloudflare's edge network BEFORE they reach your server. 
        Works with FREE Cloudflare accounts!
    </p>

    <?php if (!$enabled): ?>
    <div class="notice notice-info">
        <h3>🚀 Why Use Cloudflare Integration?</h3>
        <ul style="list-style: disc; margin-left: 20px;">
            <li><strong>Block at the Edge:</strong> Stop attacks before they consume your bandwidth</li>
            <li><strong>Save Resources:</strong> Blocked requests never hit PHP/MySQL</li>
            <li><strong>Global Network:</strong> Leverage Cloudflare's worldwide infrastructure</li>
            <li><strong>Threat Intelligence:</strong> Auto-block known malicious IPs</li>
            <li><strong>FREE Tier Compatible:</strong> No premium Cloudflare account required!</li>
        </ul>
    </div>
    <?php endif; ?>

    <form method="post" action="">
        <?php wp_nonce_field('artistpro_security_cloudflare'); ?>

        <table class="form-table">
            <tr>
                <th scope="row">
                    <label for="enabled">Enable Cloudflare Integration</label>
                </th>
                <td>
                    <label>
                        <input type="checkbox" name="enabled" id="enabled" value="1" <?php checked($enabled); ?>>
                        Sync blacklist to Cloudflare firewall
                    </label>
                    <p class="description">
                        When enabled, blocked IPs are automatically added to Cloudflare's firewall rules
                    </p>
                </td>
            </tr>

            <tr>
                <th scope="row">
                    <label for="api_token">Cloudflare API Token</label>
                </th>
                <td>
                    <input type="text" name="api_token" id="api_token" 
                           value="<?php echo esc_attr($api_token); ?>" 
                           class="regular-text" 
                           placeholder="xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx">
                    <p class="description">
                        <strong>How to get your API token:</strong><br>
                        1. Log into <a href="https://dash.cloudflare.com/profile/api-tokens" target="_blank">Cloudflare Dashboard</a><br>
                        2. Click "Create Token"<br>
                        3. Use "Edit zone DNS" template OR create custom with:<br>
                        &nbsp;&nbsp;&nbsp;- Permissions: Zone → Firewall Services → Edit<br>
                        &nbsp;&nbsp;&nbsp;- Zone Resources: Include → Specific zone → [Your domain]<br>
                        4. Copy the token and paste it here
                    </p>
                    <button type="button" class="button" id="test-cloudflare-connection">
                        Test Connection
                    </button>
                    <span id="connection-status"></span>
                </td>
            </tr>

            <tr>
                <th scope="row">
                    <label for="zone_id">Cloudflare Zone ID</label>
                </th>
                <td>
                    <input type="text" name="zone_id" id="zone_id" 
                           value="<?php echo esc_attr($zone_id); ?>" 
                           class="regular-text"
                           placeholder="xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx">
                    <p class="description">
                        Find your Zone ID in Cloudflare Dashboard → Overview (right sidebar)
                        <br><button type="button" class="button" id="fetch-zones">Auto-Detect Zones</button>
                    </p>
                    <div id="zone-list"></div>
                </td>
            </tr>

            <tr>
                <th scope="row">Auto-Sync Blacklist</th>
                <td>
                    <label>
                        <input type="checkbox" name="auto_sync" value="1" <?php checked($auto_sync); ?>>
                        Automatically sync new blacklisted IPs to Cloudflare
                    </label>
                    <p class="description">Recommended: Keep this enabled for real-time protection</p>
                    
                    <?php if ($enabled && $api_token && $zone_id): ?>
                    <p>
                        <button type="button" class="button button-secondary" id="sync-all-now">
                            🔄 Sync All Existing Blacklisted IPs Now
                        </button>
                        <span id="sync-status"></span>
                    </p>
                    <?php endif; ?>
                </td>
            </tr>

            <tr>
                <th scope="row">Threat Intelligence</th>
                <td>
                    <label>
                        <input type="checkbox" name="threat_check" value="1" <?php checked($threat_check); ?>>
                        Check Cloudflare threat score on login attempts
                    </label>
                    <p class="description">
                        Auto-block IPs with high threat scores (uses Cloudflare's global threat database)
                    </p>
                    
                    <label style="margin-top: 10px; display: block;">
                        Threat Score Threshold:
                        <input type="number" name="threat_threshold" 
                               value="<?php echo esc_attr($threat_threshold); ?>" 
                               min="0" max="100" step="1" style="width: 80px;">
                        / 100
                    </label>
                    <p class="description">
                        Block IPs with threat score above this value (recommended: 80)
                        <br>0 = Low Risk, 100 = High Risk
                    </p>
                </td>
            </tr>

            <?php if ($enabled && $api_token && $zone_id): ?>
            <tr>
                <th scope="row">Emergency: Under Attack Mode</th>
                <td>
                    <button type="button" class="button button-<?php echo $under_attack ? 'secondary' : 'primary'; ?>" 
                            id="toggle-under-attack">
                        <?php echo $under_attack ? '✅ Disable Under Attack Mode' : '🚨 Enable Under Attack Mode'; ?>
                    </button>
                    <p class="description">
                        Shows a challenge page to ALL visitors. Use during active DDoS attacks only!
                        <br><strong>Current Status:</strong> <?php echo $under_attack ? '<span style="color: red;">ACTIVE</span>' : 'Off'; ?>
                    </p>
                </td>
            </tr>

            <tr>
                <th scope="row">Cloudflare Statistics (24h)</th>
                <td>
                    <button type="button" class="button" id="refresh-cf-stats">Refresh Stats</button>
                    <div id="cloudflare-stats" style="margin-top: 15px;">
                        <em>Click "Refresh Stats" to load data...</em>
                    </div>
                </td>
            </tr>
            <?php endif; ?>
        </table>

        <p class="submit">
            <input type="submit" name="submit" class="button button-primary" value="Save Cloudflare Settings">
        </p>
    </form>
</div>

<script>
jQuery(document).ready(function($) {
    // Test connection
    $('#test-cloudflare-connection').on('click', function() {
        var $btn = $(this);
        var $status = $('#connection-status');
        
        $btn.prop('disabled', true).text('Testing...');
        $status.html('');
        
        $.post(ajaxurl, {
            action: 'test_cloudflare_connection',
            nonce: artistproSecurity.nonce
        }, function(response) {
            if (response.success) {
                $status.html('<span style="color: green;">✅ ' + response.data.message + '</span>');
            } else {
                $status.html('<span style="color: red;">❌ ' + response.data + '</span>');
            }
        }).always(function() {
            $btn.prop('disabled', false).text('Test Connection');
        });
    });

    // Fetch zones
    $('#fetch-zones').on('click', function() {
        var $btn = $(this);
        var $list = $('#zone-list');
        
        $btn.prop('disabled', true).text('Loading...');
        
        // This would need a separate AJAX handler
        $btn.prop('disabled', false).text('Auto-Detect Zones');
        $list.html('<p><em>Feature coming soon - manually enter Zone ID for now</em></p>');
    });

    // Sync all
    $('#sync-all-now').on('click', function() {
        var $btn = $(this);
        var $status = $('#sync-status');
        
        if (!confirm('Sync all blacklisted IPs to Cloudflare? This may take a moment.')) {
            return;
        }
        
        $btn.prop('disabled', true).text('Syncing...');
        $status.html('');
        
        $.post(ajaxurl, {
            action: 'sync_all_to_cloudflare',
            nonce: artistproSecurity.nonce
        }, function(response) {
            if (response.success) {
                var r = response.data;
                $status.html('<span style="color: green;">✅ Success: ' + r.success + ', Failed: ' + r.failed + ', Skipped: ' + r.skipped + '</span>');
            } else {
                $status.html('<span style="color: red;">❌ ' + response.data + '</span>');
            }
        }).always(function() {
            $btn.prop('disabled', false).text('🔄 Sync All Now');
        });
    });

    // Toggle Under Attack
    $('#toggle-under-attack').on('click', function() {
        var $btn = $(this);
        var currentlyEnabled = $btn.text().includes('Disable');
        var enable = !currentlyEnabled;
        
        if (enable && !confirm('Enable Under Attack Mode? This will challenge ALL visitors!')) {
            return;
        }
        
        $btn.prop('disabled', true).text('Please wait...');
        
        $.post(ajaxurl, {
            action: 'toggle_under_attack',
            nonce: artistproSecurity.nonce,
            enable: enable
        }, function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('Error: ' + response.data);
                $btn.prop('disabled', false);
            }
        });
    });

    // Refresh stats
    $('#refresh-cf-stats').on('click', function() {
        var $btn = $(this);
        var $stats = $('#cloudflare-stats');
        
        $btn.prop('disabled', true).text('Loading...');
        $stats.html('<em>Loading statistics...</em>');
        
        $.post(ajaxurl, {
            action: 'get_cloudflare_stats',
            nonce: artistproSecurity.nonce
        }, function(response) {
            if (response.success) {
                var s = response.data;
                var html = '<div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 15px;">';
                
                html += '<div style="background: #fff; padding: 15px; border: 1px solid #ddd; border-radius: 4px;">';
                html += '<h4 style="margin: 0 0 10px 0;">Requests</h4>';
                html += '<p><strong>Total:</strong> ' + s.requests.all.toLocaleString() + '</p>';
                html += '<p><strong>Cached:</strong> ' + s.requests.cached.toLocaleString() + '</p>';
                html += '<p><strong>Uncached:</strong> ' + s.requests.uncached.toLocaleString() + '</p>';
                html += '</div>';
                
                html += '<div style="background: #fff; padding: 15px; border: 1px solid #ddd; border-radius: 4px;">';
                html += '<h4 style="margin: 0 0 10px 0;">Bandwidth</h4>';
                html += '<p><strong>Total:</strong> ' + (s.bandwidth.all / 1024 / 1024 / 1024).toFixed(2) + ' GB</p>';
                html += '<p><strong>Saved:</strong> ' + (s.bandwidth.cached / 1024 / 1024 / 1024).toFixed(2) + ' GB</p>';
                html += '</div>';
                
                html += '<div style="background: #fff; padding: 15px; border: 1px solid #ddd; border-radius: 4px;">';
                html += '<h4 style="margin: 0 0 10px 0;">Threats Blocked</h4>';
                html += '<p><strong>Total:</strong> ' + s.threats.all.toLocaleString() + '</p>';
                html += '</div>';
                
                html += '</div>';
                
                $stats.html(html);
            } else {
                $stats.html('<span style="color: red;">❌ ' + response.data + '</span>');
            }
        }).always(function() {
            $btn.prop('disabled', false).text('Refresh Stats');
        });
    });
});
</script>

<style>
.artistpro-security-cloudflare .form-table th {
    width: 220px;
}
.artistpro-security-cloudflare .button {
    margin-right: 10px;
}
</style>
